/* It's more manageable to do:
 *    gp shimura.gp checkpol.gp
 * than to rely on $CWD being exactly where shimura.gp is to be found. */
/* \r shimura.gp */
default (parisize, 2^33);

vseq=version();
vcode=sum(i=1,3,100^(3-i)*vseq[i]);
/* ugly workaround */
if(vcode<20700, alias ("getabstime", "gettime"));

library = Str(Strexpand("$CMH_PKGLIBDIR/"), extern(Strexpand("grep dlname $CMH_PKGLIBDIR/libparsepari.la | tr \\\' \\\"")));
install ("parifopen", "vs", , library);
install ("parifclose", "v", , library);
install ("parifskipint", "vL", , library);
install ("parifreadint", "", , library);
install ("parifreadmod", "G", , library);

/* This reads one polynomial from the class polynomial file, optionally
 * reducing the result modulo p (and sending the square root in the real
 * reflex subfield to wp mod p). This returns the polynomial as it has
 * been read, and the denominator, as an integer.
 */
read_one_Hpoly(p=0,wp,skipdenom=0)=
{
   my (degree, tmp, coeffs, H, i, denom, mults);
   degree = parifreadint ();
   /* retrieve number of factors of denominator */
   tmp = parifreadint ();
   if(skipdenom,
       parifskipint (2 * tmp);
    ,
       denom=vector(tmp);
       mults=vector(tmp);
       for (i = 1, tmp,
        if(p==0,
            denom[i] = parifreadint ();
        ,
            denom[i] = Mod(parifreadint (), p);
        );
        mults[i] = parifreadint ();
       );
       denom = factorback(denom,mults);
    );
   /* read in file until the end of the first polynomial */
   H = 0;
   coeffs=vector(degree + 1, i,
      if (p == 0,
         (parifreadint () + w * parifreadint ());
      ,
         (Mod (parifreadmod (p), p) + wp * Mod (parifreadmod (p), p));
      );
   );
   H=Polrev(coeffs);
   if(skipdenom,
       return([H, degree]);
   ,
       return([H, degree, denom]);
    );
}

read_H123 (cm, p=0, wp=0, cosetnum=0, directory=".") =
/* cm a cmfield
   p an optional prime
   directory an optional string indicating in which directory the polynomial
      is to be found
   reads the polynomial file
   returns the three class polynomials (H1,H2h,H3h) in the file, with
     respect to the variable x, and each coefficient is a polynomial in w
     with w = sqrt (D0r/4) if D0r is even and sqrt(D0r) if D0r is odd,
     where D0r is the discriminant of K0r and w has a positive embedding
   If p != 0, the polynomials are reduced modulo p. p must not divide the
     any of the denominators.
*/
{
   my (K0, D, A, B, file, degree, H1, H2h, H3h, denom, tmp);
   my(coeffs, ncosets);

   K0 = cm [1];
   D = K0.disc;
   A = polcoeff (K0.pol, 1);
   B = polcoeff (K0.pol, 0);
   file = Str (directory, "/", D, "_", A, "_", B, ".pol");

   parifopen (file);

   tmp = parifreadint ();
   if (tmp != 42005 && tmp != 42006,
      error ("Unsupported file format!");
   );
   parifskipint (6);
   ncosets = parifreadint();
   if(cosetnum >= ncosets,
        error("Cannot ask for coset #", cosetnum, " as only ", ncosets, " are available");
   );
   parifskipint (1);
   if (tmp == 42005,
      parifskipint (2);
   );

   for(i=0,cosetnum,
       /* The first two ints are the coset header */
       parifskipint (2);

       gettime();
       parifskipint (2);
       tmp = read_one_Hpoly(p, wp, 1);
       H1 = tmp[1];
       if(p!=0,
         if(polcoeff(polcoeff(H1, tmp[2]),0)%p==0,
          error("p divides leading coeff");
         );
       );
       H1 /= polcoeff (H1, poldegree(H1));
       print("H1 read in: ", gettime(), " ms");

       gettime();
       parifskipint(2);
       tmp = read_one_Hpoly(p, wp, 0);
       H2h = tmp[1];
       denom = tmp[3];
       H2h /= denom;
       print("H2h read in: ", gettime(), " ms");

       gettime();
       parifskipint(2);
       tmp = read_one_Hpoly(p, wp, 0);
       H3h = tmp[1];
       denom = tmp[3];
       H3h /= denom;
       print("H3h read in: ", gettime(), " ms");
    );
   parifclose ();

   return ([H1,H2h,H3h]);
}


check_weil_ideal (cm, p) =
{
/* cm a cm field
   p a prime
   checks whether there exists a prime ideal p1 of K0r above p such that
      p splits as p1*p2 in K0r, p1 splits as q1*bar(q1) in Kr,
      the type norm of q1 is principal in K and equals (pi),
      where pi*\bar pi = p
   if yes, returns the vector [p, alpha, pi],
      where p1 is generated by p and alpha
      over the ring of integers of K0r, and pi is the type norm of q1;
   if not, returns 0
*/

   my (K0, K, unitindex, K0r, Kr, Krrel, ok, pdec, p1, q1,
       tmp, pi, eps, alpha);
   my (complex_norm = complex_norm, type_norm = type_norm);

   K0 = cm [1];
   K = cm [2];
   unitindex = cm [5];
   K0r = cm [6];
   Kr = cm [7];
   Krrel = cm [8];

   ok = 0;
   pdec = idealprimedec (Kr, p);
   for (i = 1, #pdec,
      if (pdec [i][3] == 1 && pdec [i][4] == 1,
         q1 = idealhnf (Kr, pdec [i]);
         p1 = rnfidealnormrel (Krrel, rnfidealabstorel (Krrel, Kr.zk * q1));
         tmp = bnfisprincipal (K, type_norm (cm, q1));
         if (#(tmp[1])==0 || tmp [1] == 0,
            pi = nfbasistoalg (K, tmp [2]);
            if (unitindex == 1,
               ok = 1;
               break ();
            ,
               /* check that generator has complex norm p up to norms
                  (here, squares) of units                            */
               eps = complex_norm (pi) / p;
               if (bnfisunit (K0, eps)[1] % 2 == 0,
                  ok = 1;
                  break ();
               );
            );
         );
      );
   );

   if (ok,
      alpha = lift (nfbasistoalg (K0r, idealtwoelt(K0r, p1)[2]));
         /* in terms of z */
      return ([p, alpha, pi]);
   ,
      return (0);
   );
}


find_weil_ideal (cm, pmin, pmax = -1) =
{
/* cm a cm field
   pmin an integer
   pmax an optional integer
   computes the smallest prime p >= pmin and (if pmax != -1) p <= pmax
      that passes check_weil_ideal; if one is found, the return value of
      check_weil_ideal is passed through, otherwise, 0 is returned
*/

   my (p, ntrials, res);
   my (check_weil_ideal = check_weil_ideal);
   my (startup, total, tt, lastprint);

   startup=getabstime();
   lastprint=startup;
   p = nextprime (pmin);
   ntrials = 0;
   res = 0;
   while (res === 0 && (pmax === -1 || p <= pmax),
      ntrials++;
      res = check_weil_ideal (cm, p);
      p = nextprime (p+1);
      tt=getabstime();
      if (tt >= lastprint + 10000,
        printf("Checked %d candidate Weil numbers %.1f s, %.1f ms per check (average)\n",
            ntrials, (tt-startup)/1000, (tt-startup)/ntrials);
        lastprint=tt;
      );
   );
   tt=getabstime();
   printf("Found Weil number after %d trials in %.1f s, %.1f ms per check (average)\n",
            ntrials, (tt-startup)/1000, (tt-startup)/ntrials);

   return (res);
}


find_weil_ideal_parallel (cm, pmin) =
{
/* parallelised implementation of the previous function */

   my (n, start, offset, res, v);
   my (find_weil_ideal = find_weil_ideal);

   n = default (nbthreads);
   offset = 1000;

   res = 0;
   start = pmin;
   until (res != 0,
      print (start);
      v = parvector (n, i, find_weil_ideal (cm,
                           start + (i-1) * offset, start + i * offset - 1));
      for (i = 1, n,
         if (v [i] != 0,
            res = v [i];
            break ();
         );
      );
      start += n * offset;
   );

   print (res);
   return (res);
}


weil_ideal_as_used_in_poly_files(cm, alpha) =
/* This is a conversion function only. Given alpha as returned by the
 * previous function, this returns the same object expressed as a
 * polynomial in w, where w is the element of K0r defined as w = sqrt
 * (D0r/4) if D0r is even and w = sqrt(D0r) if D0r is odd, where D0r is
 * the discriminant of K0r and w has a positive embedding.
 */
{
   my (K0r, Ar, d, pold, index);

   K0r = cm [6];
   Ar = polcoeff (K0r.pol, 1);
   d = K0r.disc;
   if (d % 2 == 0, d /= 4);
   pold = poldisc (K0r.pol);
   index = sqrtint (pold / d);
   return(subst (alpha, z, (- index * w - Ar) / 2));
}

factorization_pattern(fac)=
/* Given the factorization of a polynomial, returns a factorization
 * pattern, both in computer- and human- friendly forms
 */
{
   my (pattern,i,tdeg,d,spattern);
   tdeg=sum(i=1,#fac~,fac[i,2]*poldegree(fac[i,1]));
   pattern = vector(tdeg);
   for(i=1, #fac~,
    d=poldegree(fac[i,1]);
    pattern[d]=pattern[d]+fac[i,2];
   );
   spattern="";
   for(i=1, tdeg,
    if(pattern[i]>0,
     spattern=concat(spattern, concat(" ", concat(Str(i), concat("^", Str(pattern[i])))));
    );
   );
   /* print("factorization pattern: ", spattern); */
   return([pattern,spattern]);
}

check_completely_split(fac)=
/* Given a polynomial factorization computed by factor(),
 * check that the polynomial is completely split, or otherwise fail with
 * an error.
 */
{
   my (pat);
   pat=factorization_pattern(fac);
   if(pat[1][1] != #pat[1],
    error(concat("Wrong factorization pattern: ", pat[2]));
   );
}


invariants_streng_to_igusaclebsch(jjj)=
{
    /* Streng defines: j1, j2, j3 as:
     * I4I6'/I10, I2I4^2/I10, I4^5/I10^2
     * where I6' = (I2I4-3I6)/2
     */
    my(j1,j2,j3,i5,i2,i3,J2,J4,J6,J8,J10,I2,I4,I6,I10);

    j1=jjj[1];
    j2=jjj[2];
    j3=jjj[3];
    /* These are the [i5,i2,i3] used by Weng */
    i5=(j2-2*j1)/3;    /* I4I6/I10 */
    i2=j2^3/j3;        /* I2^3I4/I10 */
    i3=i5*i2/j2;       /* I2^2I6/I10 */

    /* These are the Igusa ones */
    J2 = 1;     /* TODO: A Mod() here ? */
    J10= 8*i5/(i2*i3);
    J4 = (-16*i5 + i3)/(24*i3);
    J6 = (80*i5*i2 - 384*i5*i3 + i2*i3)/(432*i2*i3);
    J8 = (-768*i5^2*i2 + 416*i5*i2*i3 - 1536*i5*i3^2 + i2*i3^2)/(6912*i2*i3^2);

    /* Now the Igusa-Clebsch ones */
    I2 = 8*J2;
    I4 = 4*(J2^2 - 24*J4);
    I6 = 8*(J2*(J2^2 - 20*J4) - 72*J6);
    I10= 4096*J10;

    return([I2,I4,I6,I10]);
}

invariants_igusaclebcsch_to_clebsch(III)=
{
    my(I2,I4,I6,I10,A,B,C,D);
    I2=III[1];
    I4=III[2];
    I6=III[3];
    I10=III[4];

    A =-I2/120 ;
    B =(I4+720*A^2)/6750 ;
    C =(I6-8640*A^3+108000*A*B)/202500 ;
    D =(I10+62208*A^5-972000*A^3*B-1620000*A^2*C+3037500*A*B^2+6075000*B*C)/(-4556250) ;
    return([A,B,C,D]);
}

cardona_curve_from_igusaclebsch(III)=
/* This is just to handle the special case of the automorphism group V4.
 * Untested.
 */
{
    my(CC,A,B,C,D,A11,A12,A22,A33,a111,a112,a122,a133,a222,a233,P1,P2,P3,f);
    CC=invariants_igusaclebcsch_to_clebsch(III);
    A=CC[1];
    B=CC[2];
    C=CC[3];
    D=CC[4];

    A11 = 2*C+A*B/3;
    A12 = 2*B^2/3+2*A*C/3;
    A22 = D;
    A33 = C*D-2/9*B^4+1/6*A*B*D-4/9*A*B^2*C-2/9*A^2*C^2;
    a111 = 4/75*D-8/225*B*C+4/675*A^2*C;
    a112 = (4*B^3)/675 + (8*A*B*C)/675 + (8*C^2)/225 + (2*A*D)/225;
    a122 = (2*A*B^3)/675 + (8*A^2*B*C)/2025 + (8*B^2*C)/675 + (4*A*C^2)/225 + (2*B*D)/225;
    a133 = -(A^2*B^4)/2025 + (8*B^5)/2025 - (4*A^3*B^2*C)/6075 + (14*A*B^3*C)/2025 + (2*A^2*B*C^2)/2025 + (8*B^2*C^2)/675 + (4*A*C^3)/675 + (A*B^2*D)/225 + (2*A^2*C*D)/675 + (2*B*C*D)/225 - (2*D^2)/75;
    a222 = (2*B^4)/225 + (4*A*B^2*C)/225 + (16*A^2*C^2)/2025 + (4*B*C^2)/675 - (2*C*D)/225;
    a233 = (A*B^5)/2025 + (2*A^2*B^3*C)/1215 - (2*B^4*C)/2025 + (8*A^3*B*C^2)/6075 + (2*A*B^2*C^2)/2025 + (8*A^2*C^3)/2025 - (4*B*C^3)/675 + (2*B^3*D)/675 + (A*B*C*D)/675 - (2*C^2*D)/225 - (A*D^2)/225;

    if(A22!=0,
        P1 = -2*A12-2*A22*x;
        P2 = A11-A22*x^2;
        P3 = A11+2*A12*x+A22*x^2;
        f = -A33*a111*P1^3-3*A33*a112*P1^2*P2-3*A33*a122*P1*P2^2+ 3*A22*a133*P1*P3^2-A33*a222*P2^3+3*A22*a233*P2*P3^2;
    ,
        P1 = -A11*x^2;
        P2 = -2*(A12 + A11*x);
        P3 = x*(2*A12 + A11*x);
        f = -(a111*A33*P1^3) - 3*a112*A33*P1^2*P2 - 3*a122*A33*P1*P2^2 - a222*A33*P2^3 + 3*A11*a133*P1*P3^2 + 3*A11*a233*P2*P3^2;
    );
    return([0,f]);
}

mestre_conic_and_cubic(III)=
/* Create the auxiliary cubic and conic for Mestre's algorithm */
{
    my(CC,DP,A,B,C,D,U, A11, A22, A33, A23, A31, A21, A12, A13, A32, C11, C22, C33, C23, C31, C21, C12, C13, C32, a111, a112, a113, a122, a123, a133, a222, a223, a233, a333, c111, c112, c113, c122, c123, c133, c222, c223, c233, c333,L,M,P);
    CC=invariants_igusaclebcsch_to_clebsch(III);
    DP=III[4];
    A=CC[1];
    B=CC[2];
    C=CC[3];
    D=CC[4];

    if(A!=0,U=A^6;,if(B!=0,U=B^3;,U=C^2;););

    A11 = 2*C+A*B/3 ;
    A22 = D;
    A33 = B*D/2+2*C*(B^2+A*C)/9 ;
    A23 = B*(B^2+A*C)/3+C*(2*C+A*B/3)/3 ;
    A31 = D;
    A12 = 2*(B^2+A*C)/3 ;
    A32 = A23;  A13 = A31;  A21 = A12;

    C11 = A11*U^2*DP^8/DP^11 ;
    C22 = A22*DP^10/DP^11 ;
    C33 = A33*U^8/DP^11 ;
    C23 = A23*DP^5*U^4/DP^11 ;
    C31 = A31*DP^4*U^5/DP^11 ;
    C12 = A12*U*DP^9/DP^11 ;
    C32 = C23;  C13 = C31;  C21 = C12;

    a111 = 8*(A^2*C-6*B*C+9*D)/36 ;
    a112 = 4*(2*B^3+4*A*B*C+12*C^2+3*A*D)/36 ;
    a113 = 4*(A*B^3+4*A^2*B*C/3+4*B^2*C+6*A*C^2+3*B*D)/36 ;
    a122 = a113;
    a123 = 2*(2*B^4+4*A*B^2*C+4*A^2*C^2/3+4*B*C^2+3*A*B*D+12*C*D)/36 ;
    a133 = 2*(A*B^4+4*A^2*B^2*C/3+16*B^3*C/3+26*A*B*C^2/3+ 8*C^3+3*B^2*D+2*A*C*D)/36 ;
    a222 = 4*(3*B^4+6*A*B^2*C+8*A^2*C^2/3+2*B*C^2-3*C*D)/36 ;
    a223 = 2*(-2*B^3*C/3-4*A*B*C^2/3-4*C^3+9*B^2*D+8*A*C*D)/36 ;
    a233 = 2*(B^5+2*A*B^3*C+8*A^2*B*C^2/9+2*B^2*C^2/3 -B*C*D+9*D^2)/36 ;
    a333 = 1*(-2*B^4*C-4*A*B^2*C^2-16*A^2*C^3/9-4*B*C^3/3 +9*B^3*D+12*A*B*C*D+20*C^2*D)/36 ;
    P = U^(-18)*DP^5 ;

    c111 = a111*P*U^3*DP^12 ;
    c112 = a112*P*U^2*DP^13 ;
    c113 = a113*P*U^6*DP^8 ;
    c122 = a122*P*U*DP^14 ;
    c123 = a123*P*U^5*DP^9 ;
    c133 = a133*P*U^9*DP^4 ;
    c222 = a222*P*DP^15 ;
    c223 = a223*P*U^4*DP^10 ;
    c233 = a233*P*U^8*DP^5 ;
    c333 = a333*P*U^12 ;

    L = C11*x1^2+C22*x2^2+C33*x3^2+2*C12*x1*x2+2*C13*x1*x3+2*C23*x2*x3;

    M = c111*x1^3+c222*x2^3+c333*x3^3+3*c112*x1^2*x2+3*c113*x1^2*x3+ 3*c122*x1*x2^2+3*c133*x1*x3^2+3*c233*x2*x3^2+3*c223*x2^2*x3+ 6*c123*x1*x2*x3;
     return([L,M]);
}

automorphism_group_info(III)=
/* Gets info on the automorphism group of the curve. Based on Van Wamelen
 * and Shaska.
 * This works only for prime fields with p > 5.
 */
{
    my(I2,I4,I6,I10,i1,i2,i3,i123);
    I2=III[1];
    I4=III[2];
    I6=III[3];
    I10=III[4];

    /* We may isolate one of the following groups
     * order  | what
     * 24     | see below
     * 48     | Gl_2(F_3)
     * 12     | D6
     * 10     | C10
     * 8      | D4
     * 4      | C2xC2
     * 2      | C2
     * Chance has it that the mere group order suffices in order to be
     * unambiguous...
     */

    /* C10, or 24 times more (y^2=x^5-x) if char(K) eq 5 */
    if(([I2,I4,I6]==[0,0,0]),return(10););
    i1 = I2^5/I10;
    i2 = I2^3*I4/I10;
    i3 = I2^2*I6/I10;
    i123=[i1,i2,i3];

    /* semi-direct product of C2xC6 = (Z/2)^2*Z/3 by Z/2, the
     * automorphism being any order 2 with non-trivial action both on the
     * (Z/2)^2 and the Z/3 parts.
     *
     * = aut. grp of y^2 = x^6-1
     */
    if((i123==[51200000/3,480000,148000]), return(24););

    /* Gl_2(3),  y^2 = x^5-x */
    if(i123==[400000, -20000, -2000], return(48););

    /* D4 (order 8), y^2 = x^5+x^3+t*x */
    my(d41,d42);
    d41=72000*I10 + I2^3*I4 - 82*I2*I4^2 - 3*I2^2*I6 + 240*I4*I6;
    d42=-270000*I10*I2 + 3*I2^4*I4 + 734*I2^2*I4^2 + 640*I4^3 - 9*I2^3*I6 - 4620*I2*I4*I6 + 7200*I6^2;
    if ([d41,d42] == [0,0], return(8););

    /* D6, y^2 = x^6 + x^3 + t */
    my (d61,d62);
    d61=108000*I10 + I2^3*I4 - 208*I2*I4^2 - 12*I2^2*I6 + 960*I4*I6;
    d62=5400*I10*I2 - 13*I2^2*I4^2 + 4*I4^3 + 96*I2*I4*I6 - 180*I6^2;
    if ([d61,d62] == [0,0], return(12););

    /* C2xC2 */
    my (R2);
    R2 = 125971200000*I10^3+236196*I10^2*I2^5+19245600*I10^2*I2^3*I4- 507384000*I10^2*I2*I4^2-972*I10*I2^6*I4^2-77436*I10*I2^4*I4^3+ 592272*I10*I2^2*I4^4+I2^7*I4^4-41472*I10*I4^5+78*I2^5*I4^5- 159*I2^3*I4^6+80*I2*I4^7-104976000*I10^2*I2^2*I6+ 2099520000*I10^2*I4*I6+5832*I10*I2^5*I4*I6+870912*I10*I2^3*I4^2*I6- 4743360*I10*I2*I4^3*I6-12*I2^6*I4^3*I6-1332*I2^4*I4^4*I6+ 1728*I2^2*I4^5*I6-384*I4^6*I6-8748*I10*I2^4*I6^2- 3090960*I10*I2^2*I4*I6^2+9331200*I10*I4^2*I6^2+54*I2^5*I4^2*I6^2+ 8910*I2^3*I4^3*I6^2-6048*I2*I4^4*I6^2+3499200*I10*I2*I6^3- 108*I2^4*I4*I6^3-29376*I2^2*I4^2*I6^3+6912*I4^3*I6^3+81*I2^3*I6^4+ 47952*I2*I4*I6^4-31104*I6^5;
    if(R2==0, return(4););
    /* C2, generic case */
    return(2);
}

find_point_on_conic(L)=
/* Given a trivariate homogeneous quadratic form L in variables x1,x2,x3,
 * find a point (x,y,1) on it. Currently only for finite fields. I bet
 * similar functionality exists in pari already.
 */
{
    if(type(polcoeff(L,2,x1))!="t_INTMOD",error("Currently only finite fields supported"););
    my(p,r,u);
    p=polcoeff(L,2,x1).mod;
    while(1,
        u=random(p);
        r=polrootsmod(substvec(L,[x1,x2,x3],[u,x,1]), p);
        if(#r>0,return([Mod(u,p),r[1],Mod(1,p)]););
    );
}

homogenize(f,{z=z,d=poldegree(f)})={
    /* Currently this works only for univariate polynomials */
    return(subst(f,x,x/z)*z^d);
}

curve_get_imaginary(hf)=
/* Given h and f defining the curve y^2+h(x)*y=f(x), return an imaginary
 * model of C, possibly defined over an extension field.
 */
{
    my(p,f,fa,q,d,r,homography,f2);
    if(hf[1]!=0,error("well, all the code assume char(K)>5 anyway..."););
    f=hf[2];
    p=polcoeff(f,0).mod;
    fa=factor(f);
    if(fa[1,2]!=1,error("error, curve has a finite singularity"););
    q=fa[1,1];          /* defining poly for extension (might be linear) */
    d=poldegree(q);     /* working in an extension of degree d */
    if(d==1,
        r=polrootsmod(q,p)[1];
        homography=[[z+r*x,y,x],[z,y,x-r*z]];
        f2=substvec(homogenize(f),[x,y,z],homography[1]);
        return([[0,subst(f2,z,1)],homography]);
    ,
        /* Need to work in an extension. Not that it's harder, but my
         * level of acquaintance with gp's idiosyncrasies is not
         * sufficient.  */
        my(pat);
        pat=factorization_pattern(factor(f));
        warning("Present code currently can't compute imaginary model in an extension ; pattern of f: ", pat[2]);
        return(0);
    );
}

curve_from_igusaclebsch(III)=
/* give an equation for a genus 2 hyperelliptic curve having the given
 * invariants. Mestre's algorithm. */
{
    my(I2,I4,I6,I10,p,g,z);
    I2=III[1];
    I4=III[2];
    I6=III[3];
    I10=III[4];
    p=I10.mod;
    g=automorphism_group_info(III);
    if(g!=2,warning("curve_from_igusaclebsch untested beyond the case of trivial automorphism groups"););
    if(g==240,return([0,x^5-x]););/* characteristic 5 only */
    if(g==48,return([0,x^5-x]););
    if(g==10,return([0,x^5-1]););
    if(g==24,return([0,x^6-1]););
    if(g==8,
        z=(3*(36000*I10 + I2^3*I4 + 44*I2*I4^2 + 6*I2^2*I6 - 480*I4*I6))/ (10800000*I10 + 9*I2^5 + 700*I2^3*I4 - 12400*I2*I4^2 - 3600*I2^2*I6 + 48000*I4*I6);
        return([0,x^5+x^3+z*x]);
    );
    if(g==12,
        z = (2*I2*I4 - 20*I6)/(3*I2^3 + 140*I2*I4 - 800*I6);
        return([0,x^6 + x^3 + z]);
    );
    if(g==4,return(cardona_curve_from_igusaclebsch(III)););
    my(LM,L,M,P,u,v,Ltm,a,b,vL,vM);
    LM=mestre_conic_and_cubic(III);
    L=LM[1];
    M=LM[2];
    P=find_point_on_conic(L);
    u=P[1];
    v=P[2];
    /* Now we have a polynomial in t, for each slope value m.
     * The intersection points, given m, are the roots of Ltm in t. Of
     * course (u,v) should be on the conic, thus we expect Ltm to be a
     * multiple of t */
    Ltm=subst(subst(subst(L,x1,u+m*t),x2,v+t),x3,1);
    if(polcoeff(Ltm,0,t)!=0,error("Argh"););
    /* Express the other t from m, as t=-b/a */
    b=polcoeff(Ltm,1,t);
    a=polcoeff(Ltm,2,t);
    /* Evaluate L at [u+m*-b/a, v+-b/a, 1] == [au-bm,av-b,a] -> get 0 */
    vL=subst(subst(subst(L,x1,a*u-b*m),x2,a*v-b),x3,a);
    vM=subst(subst(subst(M,x1,a*u-b*m),x2,a*v-b),x3,a);
    if(vL!=0,error("Argh"););
    /* return (subst(lift(vM),m,x)); */
    my(hf,hf2);
    hf = [0,subst(vM,m,x)];
    hf2 = curve_get_imaginary(hf);
    if(hf2!=0, hf=hf2[1];);
    print("Taking C: y^2+",lift(hf[1]),"*y=",lift(hf[2]));
    return(hf);
}


polmod(f,p)={return(sum(i=0,poldegree(f),Mod(polcoeff(f,i),p)*x^i));}

divisor_add(hf,D1,D2)=
/* Cantor addition on the jacobian of y^2+h(x)y=f(x). Stay naive */
{
    my(h,f,p,u1,u2,v1,v2,be,t1,t2,d0,be2,s1,s2,s3,d,u,v);
    h=hf[1];
    f=hf[2];
    if(poldegree(f)>5,error("This code assumes genus 2 imaginary model"););
    p=polcoeff(f,0).mod;
    u1=polmod(D1[1]*x^0,p);
    u2=polmod(D2[1]*x^0,p);
    v1=polmod(D1[2]*x^0,p);
    v2=polmod(D2[2]*x^0,p);
    /* work around pari-2.5.0 bug :-(( */
    if(u1==u2,be=[0,1,u1],be=bezout(u1,u2));
    t1=be[1];
    t2=be[2];
    d0=be[3];
    be2=bezout(d0,v1+v2+h);
    s1=t1*be2[1];
    s2=t2*be2[1];
    s3=be2[2];
    d=be2[3];
    u=u1*u2/d^2;
    v=((s1*u1*v2+s2*u2*v1+s3*(v1*v2+f))/d) % u;
    while(poldegree(u)>2, u=(f-h*v-v^2)/u; v=(-h-v)%u;);
    if(u!=0,u/=polcoeff(u,poldegree(u)););
    return(lift([u,v]));
}

divisor_random(hf)=
/* pick a random divisor on the jacobian of y^2+h(x)y=f(x). We do not
 * claim to be uniform. In fact, presently, we're even downright stupid,
 * and generate a completely split divisor. It misses half of the story.
 */
{
    my(f,p,x0,y0,x1,y1);
    if(hf[1]!=0,error("well, all the code assume char(K)>5 anyway..."););
    f=hf[2];
    p=polcoeff(f,0).mod;
    while(1,
        x0=random(Mod(0,p));
        if(issquare(subst(f,x,x0),&y0),break;);
    );
    while(1,
        x1=random(Mod(0,p));
        if(x1!=x0 && issquare(subst(f,x,x1),&y1),break;);
    );
    return(lift([(x-x0)*(x-x1),y0*(x-x1)/(x0-x1)+y1*(x-x0)/(x1-x0)]));
}


divisor_homogenize(D)=
/* Homogenize the Mumford divisor representing the ideal [D[1],y-D[2]] as
 * a function on the weighted projective space in variables [x,y,z] with
 * weights [1,3,1].
 */
{
    return([homogenize(D[1],z),homogenize(D[2],z,3)]);
}

divisor_apply_homography(D,phi)=
/* Given a linear transformation phi on P^2 (well, currently really only
 * P^1, since we require y be unchanged), transform the divisor D ono the
 * curve defined by y^2-f into the divisor D\circ\phi defined on
 * fx=(y^2-f)\circ phi. Again, because of weighted projective space, we
 * have the modding out by Dx1[1] in the end.
 */
{
    my(Dx,Dx1);
    Dx=substvec(divisor_homogenize(D),[x,y,z],phi);
    Dx1=subst(Dx,z,1);
    Dx1[2]%=Dx1[1];
    return(Dx1);
}

divisor_multiply(hf,n,D)=
/* compute n times D.
 *
 * [disabled --
 * Since we don't have working arithmetic on real
 * models, we map the whole thing to an imaginary model, and back.
 * Presently, we don't handle the case where the imaginary model is
 * defined over an extension.
 * ] -- [curve is chosen imaginary from the start, now. Won't always
 * work, of course, but at least it doesn't propagate the mess here]
 */
{
    my(p,m,r,mapback);
    p=polcoeff(hf[2],0).mod;
    /*
    mapback=[x,y,z];
    if(poldegree(hf[2])==6,
        my(immodel);
        immodel=curve_get_imaginary(hf);
        print("Applying homography ", lift(immodel[2][1]), " to imaginary model\n");
        hf=immodel[1];
        mapback=immodel[2][2];
        D=divisor_apply_homography(D,immodel[2][1]);
    );
    */

    if((D[2]^2-hf[2])%D[1] != 0, error("Divisor not on curve"););

    r=[1,0];
    m=1;
    while(m<n,m*=2;);
    while(m>0,
        r=divisor_add(hf,r,r);
        if(n>=m,
          r=divisor_add(hf,r,D);
          n-=m;
        );
        m=divrem(m,2)[1];
    );
    /* r=divisor_apply_homography(r,mapback); */
    return(r);
}

guess_jacobian_cardinal_from_random_points(hf, expect)=
/* Given h and f defining the hyperelliptic curve C: y^2+h(x)y=f(x), try to
 * prove the correct value for the cardinal of the Jacobian of C, under
 * the assumption that it belongs to the set of expected value provided
 * as argument.
 *
 * -1 is returned on failure.
 */
{
   my(ntrials,next_expect,D,n);
   ntrials=0;

   while(#expect>1 || (#expect == 1 && ntrials<5),
       D=divisor_random(hf);
       ntrials+=1;
       next_expect=[];
       for(i=1,#expect,
           n=expect[i];
           if(divisor_multiply(hf,n,D)==[1,0],
             next_expect=concat(next_expect,expect[i]);
           );
       );
       expect=next_expect;
    );
   if(#expect==0, return(-1););
   return(expect[1]);
}

find_one_invariant_triple_avoiding_multiple_roots(HHs, fac, p, i0=1)=
/* Given a triple of class polynomials H1, H2hat, H3hat modulo p, and the
 * factorization of H1 (modulo p), return an arbitrary point in the
 * algebraic set defined by these class polynomials, making sure that we
 * pick one which does not correspond to a multiple root of H1.
 *
 * return 0 on failure.
 */
{
   my(H1p,H2hp,H3hp,j1,j2,j3,delta);
   H1p=HHs[1];
   H2hp=HHs[2];
   H3hp=HHs[3];
   for(i=i0, #fac~,
    if(fac[i,2]==1,
     /* We've already made sure it's degree 1 */
     j1=polrootsmod(fac[i,1],p)[1];
     delta=subst(deriv(H1p,x),x,j1);
     j2=subst(H2hp,x,j1)/delta;
     j3=subst(H3hp,x,j1)/delta;
     if ([j1,j2,j3] == 0,
        /* avoid triple with I4==0 */
        print("Not considering invariant triple [0,0,0]");
     ,
        return ([j1,j2,j3]);
     );
    );
   );
   return (0);
}

/* H must be a polynomial in x with coefficients mod p. This counts the
 * number of roots in GF(p) (as opposed to roots over extension fields). */
number_of_rational_roots(H,with_multiplicities=1)=
{
    if(type(H)!="t_POL",error("H must be t_POL"););
    if(type(polcoeff(H,0))!="t_INTMOD",error("H must have coefficients of type t_INTMOD"););
    my (p,splitpart,tail,nroots);
    p=polcoeff(H,0).mod;
    splitpart=gcd(lift(Mod(x,H)^p)-x,H);
    nroots=poldegree(splitpart);
    /* recurse over multiple roots */
    if(with_multiplicities && nroots>0 && nroots < poldegree(H),
        tail=H/splitpart;
        nroots+=number_of_rational_roots(tail);
    );
    return(nroots);
}

/* H must be a polynomial in x with coefficients mod p. This returns the
 * factor of H whose roots are the roots of H mod p, and all
 * multiplicities are capped to 1.
 */
pol_prod_of_single_roots(H)=
{
    if(type(H)!="t_POL",error("H must be t_POL"););
    if(type(polcoeff(H,0))!="t_INTMOD",error("H must have coefficients of type t_INTMOD"););
    my (p,M,S);
    p=polcoeff(H,0).mod;
    S=gcd(lift(Mod(x,H)^p)-x,H);
    M=H/S;
    S=S/gcd(M, S);
    return(S);
}

/* H must be a polynomial in x with coefficients mod p. This returns a
 * factor of H mod p, with single roots modulo p, and with the constraint
 * that the degree of the result does not exceed the parameter "atmost".
 */
several_polrootsmod(H, atmost=100)=
{
    if(type(H)!="t_POL",error("H must be t_POL"););
    if(type(polcoeff(H,0))!="t_INTMOD",error("H must have coefficients of type t_INTMOD"););
    my (p,r,C,tt0,tt);
    p=polcoeff(H,0).mod;
    if(poldegree(H) < atmost,
        return(H);
    );
    while(1,
        tt0=getabstime();
        r=random(p);
        C=gcd(lift(Mod(x+r,H)^((p-1)/2))-1,H);
        tt=(getabstime()-tt0)/1000;
        if(tt>1,
        printf("Doing pow+gcd on polynomial of degree %d took %.1fs; deg result=%d\n", poldegree(H), tt, poldegree(C));
        );
        if(poldegree(C)>0, break;);
    );
    if(poldegree(C) > poldegree(H)/2, C=H/C;);
    return(several_polrootsmod(C, atmost));
}

/* H must be a polynomial in x with coefficients mod p. This returns
 * *one* random root of H in GF(p).
 */
one_polrootmod(H)=
{
    return(polrootsmod(several_polrootsmod(H,1),polcoeff(H,0).mod)[1]);
}    



checkpol (cm, p0=1, parallel=0, cosetnum=0, directory=".") =
/* cm a cm field
 * p0 an optional integer
 * parallel an optional integer
 * directory an optional string indicating in which directory the polynomial
 *    is to be found
 * Determines a curve corresponding to a triple of roots of the class
 *    polynomials modulo a prime >= p0; errors out if this is not possible,
 *    and returns a vector with the prime, the triple [j1, j2, j3],
 *    the curve equation (precisely, the f of Y^2 = f (X)) and the
 *    cardinality of its Jacobian if everything goes well.
 * If parallel is TRUE, uses the parallel version.
 */
{
   my (K0r, p, pi, wp, H1p, H2hp, H3hp, tmp, alpha, HHH, fac, jjj, hf, e);
   my (startup, total, tt);

   K0r = cm [6];

   startup=getabstime();

   tmp = if (parallel,
            find_weil_ideal_parallel (cm, p0);
         ,
            find_weil_ideal (cm, p0);
         );
   p = tmp [1];
   alpha = weil_ideal_as_used_in_poly_files(cm, tmp[2]);
   pi = tmp [3];
   print ("Choosing prime p=", p);
   print ("pi=", pi);

   /* alpha defines exactly how we must reduce */
   /* If parsing of polynomial files within pari becomes an issue, we may
    * delegate it to the C program (in fact, I have the impression that
    * pari is fast enough)
    */
   wp = - Mod (polcoeff (alpha, 0) / polcoeff (alpha, 1), p);

   print("Projecting class polynomials to GF(p) with w->", lift(wp));

   HHH = read_H123 (cm, p, wp, cosetnum, directory);
   H1p = HHH[1];
   H2hp = HHH[2];
   H3hp = HHH[3];

   print ("(H1,H2h,H3h) read; class number is ", poldegree(H1p), ".");

   tt=getabstime();
   print("Number of roots of H1 mod p (with multiplicities): ", number_of_rational_roots(H1p));
   tt=(getabstime()-tt)/1000;
   if(tt > 1,
    printf("Counting roots took %.1f s\n", tt);
   );

   print("Limiting to 100 candidate roots");
   fac=factor(several_polrootsmod(H1p, 100));
   /*
   fac = factor(H1p);
   */
   check_completely_split(fac);

   /* Example w/ multiple roots: A=35, B=65 p=419 */
   jjj = find_one_invariant_triple_avoiding_multiple_roots([H1p,H2hp,H3hp],fac,p);
   if(jjj===0, error("Cannot find a non-multiple invariant triple"););
   print("Taking invariants ", lift(jjj));


   /* This is just for checking the fact that pi has the correct
    * _relative_ norm, which ought to be p. In fact we don't need it:
      Krel = cm[3];
      pirel=rnfeltabstorel(Krel,pi);
      norm(pirel)==p;
    */

   /* we get only f (we don't support characteristic <=5 anyway) */
   hf=curve_from_igusaclebsch(invariants_streng_to_igusaclebsch(jjj));

   /* The expected cardinal of the Jacobian is norm(1 \pm pi) (the
    * absolute norm).  */
   print("As per Norm(1+pi) and Norm(1-pi), candidates for #J are: ",
        [norm(1-pi),norm(1+pi)]);
   if (poldegree(hf[2])>5,
    print("\nWe could not find a curve with imaginary model for this polynomial, and thus we cannot check. This is not a bug, but a functionality lack in the code.  Magma can be used to check that the Jacobian cardinal corresponds to one of the possible values above.  Here is some example Magma code for doing that:\n");
        printf("/* magma check %d %d */\n", polcoeff(cm[1].pol,1), polcoeff(cm[1].pol,0));
        printf("p:=%d;\nKP<x>:=PolynomialRing(GF(p));\n", p);
        print("J:=Jacobian(HyperellipticCurve(", lift(hf[2]), "));");
        printf("if IsZero(%d*Random(J)) or IsZero(%d*Random(J)) then print \"check: ok %d, %d (magma)\"; else print \"check: FAIL %d, %d (magma)\"; end if;\n", norm(1-pi),norm(1+pi), polcoeff(cm[1].pol,1), polcoeff(cm[1].pol,0), polcoeff(cm[1].pol,1), polcoeff(cm[1].pol,0));
        quit(2);
   );
   e=guess_jacobian_cardinal_from_random_points(hf, [norm(1-pi),norm(1+pi)]);

   if(e<0,
    print("ERROR: Cannot obtain correct jacobian cardinal.");
    quit(1);
   ,
    print("#J=", e);
   );

   total=getabstime()-startup;
   printf ("Time for checking:        %.1f\n", gettime () / 1000.0);
   return([p,lift(jjj),lift(hf[2]),e]);
}

/*
{
A = 111;
B = 631;
print (checkpol (init_cmfield (A, B), 1, 0));
}
*/
