# Distributed under the OSI-approved BSD 3-Clause License.
# See accompanying file Copyright.txt or https://cmake.org/licensing
# for details.

#[================================================================[.rst:
FindDiscordRPC
-------

Finds the discord-rpc library.

Imported Targets
^^^^^^^^^^^^^^^^

This module provides the following imported targets, if found:

``discord-rpc::discord-rpc``
  The discord-rpc library

Result Variables
^^^^^^^^^^^^^^^^

This will define the following variables:

``DiscordRPC_FOUND``
  True if the system has the discord-rpc library.
``DiscordRPC_VERSION``
  The version of the discord-rpc library which was found.
``DiscordRPC_INCLUDE_DIRS``
  Include directories needed to use discord-rpc.
``DiscordRPC_LIBRARIES``
  Libraries needed to link to discord-rpc.

Cache Variables
^^^^^^^^^^^^^^^

The following cache variables may also be set:

``DiscordRPC_INCLUDE_DIR``
  The directory containing ``discord_rpc.h``.
``DiscordRPC_LIBRARY``
  The path to the discord-rpc library.

#]================================================================]

# The following code was adapted from the instructions provided at
# https://cmake.org/cmake/help/latest/manual/cmake-developer.7.html

find_package(PkgConfig QUIET)
if(PkgConfig_FOUND)
  pkg_check_modules(PC_DiscordRPC QUIET discord-rpc)
  find_path(DiscordRPC_INCLUDE_DIR
    NAMES discord_rpc.h
    PATHS ${PC_DiscordRPC_INCLUDE_DIRS}
  )
  find_library(DiscordRPC_LIBRARY
    NAMES discord-rpc
    PATHS ${PC_DiscordRPC_LIBRARY_DIRS}
  )
else()
  find_path(DiscordRPC_INCLUDE_DIR NAMES discord_rpc.h)
  find_library(DiscordRPC_LIBRARY  NAMES discord-rpc)
endif(PkgConfig_FOUND)

# Set library version
if(PC_DiscordRPC_VERSION)
  set(DiscordRPC_VERSION ${PC_DiscordRPC_VERSION})
else()
  set(DiscordRPC_VERSION 3.4.0)
endif()
if(${DiscordRPC_VERSION} MATCHES [[([0-9]+)(\.([0-9]+)(\.([0-9]+))?)?]])
  set(DiscordRPC_VERSION_MAJOR "${CMAKE_MATCH_1}")
  if(CMAKE_MATCH_2)
    set(DiscordRPC_VERSION_MINOR "${CMAKE_MATCH_3}")
  endif()
  if(CMAKE_MATCH_4)
    set(DiscordRPC_VERSION_PATCH "${CMAKE_MATCH_5}")
  endif()
endif()

include(FindPackageHandleStandardArgs)
find_package_handle_standard_args(DiscordRPC
  FOUND_VAR DiscordRPC_FOUND
  REQUIRED_VARS
    DiscordRPC_LIBRARY
    DiscordRPC_INCLUDE_DIR
  VERSION_VAR DiscordRPC_VERSION
)

if(DiscordRPC_FOUND AND NOT TARGET discord-rpc::discord-rpc)
  add_library(discord-rpc::discord-rpc UNKNOWN IMPORTED)
  set_target_properties(discord-rpc::discord-rpc PROPERTIES
    IMPORTED_LOCATION "${DiscordRPC_LIBRARY}"
    INTERFACE_COMPILE_OPTIONS "${PC_DiscordRPC_CFLAGS_OTHER}"
    INTERFACE_INCLUDE_DIRECTORIES "${DiscordRPC_INCLUDE_DIR}"
  )
endif()

mark_as_advanced(
  DiscordRPC_INCLUDE_DIR
  DiscordRPC_LIBRARY
)
