! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
#define ESMF_FILENAME "ESMF_FieldEmpty.F90"
!==============================================================================
!
! ESMF FieldEmpty module
module ESMF_FieldEmptyMod
!
!==============================================================================
!
! This file contains methods related to partially created Fields.
!
!------------------------------------------------------------------------------

#include "ESMF.h"

!------------------------------------------------------------------------------
! !USES:
  use ESMF_UtilTypesMod
  use ESMF_BaseMod
  use ESMF_InitMacrosMod
  use ESMF_LogErrMod
  use ESMF_ArraySpecMod
  use ESMF_LocalArrayMod
  use ESMF_DistGridMod
  use ESMF_StaggerLocMod
  use ESMF_GridMod
  use ESMF_XGridMod
  use ESMF_MeshMod
  use ESMF_LocStreamMod
  use ESMF_GeomMod
  use ESMF_ArrayMod
  use ESMF_FieldMod
  use ESMF_FieldGetMod
  use ESMF_VMMod

  implicit none

!------------------------------------------------------------------------------
! !PRIVATE TYPES:
  private

!------------------------------------------------------------------------------
! !PUBLIC MEMBER FUNCTIONS:
  public ESMF_FieldEmptyCreate
  public ESMF_FieldEmptyReset
  public ESMF_FieldEmptySet
  public ESMF_FieldEmptyComplete
  public ESMF_FieldConstructIANew ! For internal ESMF use only
  public ESMF_FieldConstructIANewArray! For internal ESMF use only

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
  character(*), parameter, private :: version = &
    '$Id$'

!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_FieldEmptySet - Set a Geom object in an empty Field
!
! !INTERFACE:
  interface ESMF_FieldEmptySet

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldEmptySetGeom
    module procedure ESMF_FieldEmptySetGrid
    module procedure ESMF_FieldEmptySetMesh
    module procedure ESMF_FieldEmptySetLocStream
    module procedure ESMF_FieldEmptySetXGrid

! !DESCRIPTION:
! This interface sets a {\tt ESMF\_Geom} object in an empty {\tt ESMF\_Field}.
! The geom object can take {\tt ESMF\_Grid}, or {\tt ESMF\_LocStream}, or {\tt ESMF\_Mesh},
! or {\tt ESMF\_XGrid}.
  end interface
!EOPI

!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_FieldEmptyComplete - Completes a Field started with FieldEmptyCreate
!
! !INTERFACE:
  interface ESMF_FieldEmptyComplete

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldEmptyCompAS
    module procedure ESMF_FieldEmptyCompTK
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyComp1DI1 
 module procedure ESMF_FieldEmptyComp2DI1 
 module procedure ESMF_FieldEmptyComp3DI1 
 module procedure ESMF_FieldEmptyComp4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyComp5DI1 
 module procedure ESMF_FieldEmptyComp6DI1 
 module procedure ESMF_FieldEmptyComp7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyComp1DI2 
 module procedure ESMF_FieldEmptyComp2DI2 
 module procedure ESMF_FieldEmptyComp3DI2 
 module procedure ESMF_FieldEmptyComp4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyComp5DI2 
 module procedure ESMF_FieldEmptyComp6DI2 
 module procedure ESMF_FieldEmptyComp7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyComp1DI4 
 module procedure ESMF_FieldEmptyComp1DI8 
 module procedure ESMF_FieldEmptyComp1DR4 
 module procedure ESMF_FieldEmptyComp1DR8 
 module procedure ESMF_FieldEmptyComp2DI4 
 module procedure ESMF_FieldEmptyComp2DI8 
 module procedure ESMF_FieldEmptyComp2DR4 
 module procedure ESMF_FieldEmptyComp2DR8 
 module procedure ESMF_FieldEmptyComp3DI4 
 module procedure ESMF_FieldEmptyComp3DI8 
 module procedure ESMF_FieldEmptyComp3DR4 
 module procedure ESMF_FieldEmptyComp3DR8 
 module procedure ESMF_FieldEmptyComp4DI4 
 module procedure ESMF_FieldEmptyComp4DI8 
 module procedure ESMF_FieldEmptyComp4DR4 
 module procedure ESMF_FieldEmptyComp4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyComp5DI4 
 module procedure ESMF_FieldEmptyComp5DI8 
 module procedure ESMF_FieldEmptyComp5DR4 
 module procedure ESMF_FieldEmptyComp5DR8 
 module procedure ESMF_FieldEmptyComp6DI4 
 module procedure ESMF_FieldEmptyComp6DI8 
 module procedure ESMF_FieldEmptyComp6DR4 
 module procedure ESMF_FieldEmptyComp6DR8 
 module procedure ESMF_FieldEmptyComp7DI4 
 module procedure ESMF_FieldEmptyComp7DI8 
 module procedure ESMF_FieldEmptyComp7DR4 
 module procedure ESMF_FieldEmptyComp7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompPtr1DI1 
 module procedure ESMF_FieldEmptyCompPtr2DI1 
 module procedure ESMF_FieldEmptyCompPtr3DI1 
 module procedure ESMF_FieldEmptyCompPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompPtr5DI1 
 module procedure ESMF_FieldEmptyCompPtr6DI1 
 module procedure ESMF_FieldEmptyCompPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompPtr1DI2 
 module procedure ESMF_FieldEmptyCompPtr2DI2 
 module procedure ESMF_FieldEmptyCompPtr3DI2 
 module procedure ESMF_FieldEmptyCompPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompPtr5DI2 
 module procedure ESMF_FieldEmptyCompPtr6DI2 
 module procedure ESMF_FieldEmptyCompPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompPtr1DI4 
 module procedure ESMF_FieldEmptyCompPtr1DI8 
 module procedure ESMF_FieldEmptyCompPtr1DR4 
 module procedure ESMF_FieldEmptyCompPtr1DR8 
 module procedure ESMF_FieldEmptyCompPtr2DI4 
 module procedure ESMF_FieldEmptyCompPtr2DI8 
 module procedure ESMF_FieldEmptyCompPtr2DR4 
 module procedure ESMF_FieldEmptyCompPtr2DR8 
 module procedure ESMF_FieldEmptyCompPtr3DI4 
 module procedure ESMF_FieldEmptyCompPtr3DI8 
 module procedure ESMF_FieldEmptyCompPtr3DR4 
 module procedure ESMF_FieldEmptyCompPtr3DR8 
 module procedure ESMF_FieldEmptyCompPtr4DI4 
 module procedure ESMF_FieldEmptyCompPtr4DI8 
 module procedure ESMF_FieldEmptyCompPtr4DR4 
 module procedure ESMF_FieldEmptyCompPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompPtr5DI4 
 module procedure ESMF_FieldEmptyCompPtr5DI8 
 module procedure ESMF_FieldEmptyCompPtr5DR4 
 module procedure ESMF_FieldEmptyCompPtr5DR8 
 module procedure ESMF_FieldEmptyCompPtr6DI4 
 module procedure ESMF_FieldEmptyCompPtr6DI8 
 module procedure ESMF_FieldEmptyCompPtr6DR4 
 module procedure ESMF_FieldEmptyCompPtr6DR8 
 module procedure ESMF_FieldEmptyCompPtr7DI4 
 module procedure ESMF_FieldEmptyCompPtr7DI8 
 module procedure ESMF_FieldEmptyCompPtr7DR4 
 module procedure ESMF_FieldEmptyCompPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompGrid1DI1 
 module procedure ESMF_FieldEmptyCompGrid2DI1 
 module procedure ESMF_FieldEmptyCompGrid3DI1 
 module procedure ESMF_FieldEmptyCompGrid4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGrid5DI1 
 module procedure ESMF_FieldEmptyCompGrid6DI1 
 module procedure ESMF_FieldEmptyCompGrid7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompGrid1DI2 
 module procedure ESMF_FieldEmptyCompGrid2DI2 
 module procedure ESMF_FieldEmptyCompGrid3DI2 
 module procedure ESMF_FieldEmptyCompGrid4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGrid5DI2 
 module procedure ESMF_FieldEmptyCompGrid6DI2 
 module procedure ESMF_FieldEmptyCompGrid7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompGrid1DI4 
 module procedure ESMF_FieldEmptyCompGrid1DI8 
 module procedure ESMF_FieldEmptyCompGrid1DR4 
 module procedure ESMF_FieldEmptyCompGrid1DR8 
 module procedure ESMF_FieldEmptyCompGrid2DI4 
 module procedure ESMF_FieldEmptyCompGrid2DI8 
 module procedure ESMF_FieldEmptyCompGrid2DR4 
 module procedure ESMF_FieldEmptyCompGrid2DR8 
 module procedure ESMF_FieldEmptyCompGrid3DI4 
 module procedure ESMF_FieldEmptyCompGrid3DI8 
 module procedure ESMF_FieldEmptyCompGrid3DR4 
 module procedure ESMF_FieldEmptyCompGrid3DR8 
 module procedure ESMF_FieldEmptyCompGrid4DI4 
 module procedure ESMF_FieldEmptyCompGrid4DI8 
 module procedure ESMF_FieldEmptyCompGrid4DR4 
 module procedure ESMF_FieldEmptyCompGrid4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGrid5DI4 
 module procedure ESMF_FieldEmptyCompGrid5DI8 
 module procedure ESMF_FieldEmptyCompGrid5DR4 
 module procedure ESMF_FieldEmptyCompGrid5DR8 
 module procedure ESMF_FieldEmptyCompGrid6DI4 
 module procedure ESMF_FieldEmptyCompGrid6DI8 
 module procedure ESMF_FieldEmptyCompGrid6DR4 
 module procedure ESMF_FieldEmptyCompGrid6DR8 
 module procedure ESMF_FieldEmptyCompGrid7DI4 
 module procedure ESMF_FieldEmptyCompGrid7DI8 
 module procedure ESMF_FieldEmptyCompGrid7DR4 
 module procedure ESMF_FieldEmptyCompGrid7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompGridPtr1DI1 
 module procedure ESMF_FieldEmptyCompGridPtr2DI1 
 module procedure ESMF_FieldEmptyCompGridPtr3DI1 
 module procedure ESMF_FieldEmptyCompGridPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGridPtr5DI1 
 module procedure ESMF_FieldEmptyCompGridPtr6DI1 
 module procedure ESMF_FieldEmptyCompGridPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompGridPtr1DI2 
 module procedure ESMF_FieldEmptyCompGridPtr2DI2 
 module procedure ESMF_FieldEmptyCompGridPtr3DI2 
 module procedure ESMF_FieldEmptyCompGridPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGridPtr5DI2 
 module procedure ESMF_FieldEmptyCompGridPtr6DI2 
 module procedure ESMF_FieldEmptyCompGridPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompGridPtr1DI4 
 module procedure ESMF_FieldEmptyCompGridPtr1DI8 
 module procedure ESMF_FieldEmptyCompGridPtr1DR4 
 module procedure ESMF_FieldEmptyCompGridPtr1DR8 
 module procedure ESMF_FieldEmptyCompGridPtr2DI4 
 module procedure ESMF_FieldEmptyCompGridPtr2DI8 
 module procedure ESMF_FieldEmptyCompGridPtr2DR4 
 module procedure ESMF_FieldEmptyCompGridPtr2DR8 
 module procedure ESMF_FieldEmptyCompGridPtr3DI4 
 module procedure ESMF_FieldEmptyCompGridPtr3DI8 
 module procedure ESMF_FieldEmptyCompGridPtr3DR4 
 module procedure ESMF_FieldEmptyCompGridPtr3DR8 
 module procedure ESMF_FieldEmptyCompGridPtr4DI4 
 module procedure ESMF_FieldEmptyCompGridPtr4DI8 
 module procedure ESMF_FieldEmptyCompGridPtr4DR4 
 module procedure ESMF_FieldEmptyCompGridPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGridPtr5DI4 
 module procedure ESMF_FieldEmptyCompGridPtr5DI8 
 module procedure ESMF_FieldEmptyCompGridPtr5DR4 
 module procedure ESMF_FieldEmptyCompGridPtr5DR8 
 module procedure ESMF_FieldEmptyCompGridPtr6DI4 
 module procedure ESMF_FieldEmptyCompGridPtr6DI8 
 module procedure ESMF_FieldEmptyCompGridPtr6DR4 
 module procedure ESMF_FieldEmptyCompGridPtr6DR8 
 module procedure ESMF_FieldEmptyCompGridPtr7DI4 
 module procedure ESMF_FieldEmptyCompGridPtr7DI8 
 module procedure ESMF_FieldEmptyCompGridPtr7DR4 
 module procedure ESMF_FieldEmptyCompGridPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompLS1DI1 
 module procedure ESMF_FieldEmptyCompLS2DI1 
 module procedure ESMF_FieldEmptyCompLS3DI1 
 module procedure ESMF_FieldEmptyCompLS4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompLS5DI1 
 module procedure ESMF_FieldEmptyCompLS6DI1 
 module procedure ESMF_FieldEmptyCompLS7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompLS1DI2 
 module procedure ESMF_FieldEmptyCompLS2DI2 
 module procedure ESMF_FieldEmptyCompLS3DI2 
 module procedure ESMF_FieldEmptyCompLS4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompLS5DI2 
 module procedure ESMF_FieldEmptyCompLS6DI2 
 module procedure ESMF_FieldEmptyCompLS7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompLS1DI4 
 module procedure ESMF_FieldEmptyCompLS1DI8 
 module procedure ESMF_FieldEmptyCompLS1DR4 
 module procedure ESMF_FieldEmptyCompLS1DR8 
 module procedure ESMF_FieldEmptyCompLS2DI4 
 module procedure ESMF_FieldEmptyCompLS2DI8 
 module procedure ESMF_FieldEmptyCompLS2DR4 
 module procedure ESMF_FieldEmptyCompLS2DR8 
 module procedure ESMF_FieldEmptyCompLS3DI4 
 module procedure ESMF_FieldEmptyCompLS3DI8 
 module procedure ESMF_FieldEmptyCompLS3DR4 
 module procedure ESMF_FieldEmptyCompLS3DR8 
 module procedure ESMF_FieldEmptyCompLS4DI4 
 module procedure ESMF_FieldEmptyCompLS4DI8 
 module procedure ESMF_FieldEmptyCompLS4DR4 
 module procedure ESMF_FieldEmptyCompLS4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompLS5DI4 
 module procedure ESMF_FieldEmptyCompLS5DI8 
 module procedure ESMF_FieldEmptyCompLS5DR4 
 module procedure ESMF_FieldEmptyCompLS5DR8 
 module procedure ESMF_FieldEmptyCompLS6DI4 
 module procedure ESMF_FieldEmptyCompLS6DI8 
 module procedure ESMF_FieldEmptyCompLS6DR4 
 module procedure ESMF_FieldEmptyCompLS6DR8 
 module procedure ESMF_FieldEmptyCompLS7DI4 
 module procedure ESMF_FieldEmptyCompLS7DI8 
 module procedure ESMF_FieldEmptyCompLS7DR4 
 module procedure ESMF_FieldEmptyCompLS7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompLSPtr1DI1 
 module procedure ESMF_FieldEmptyCompLSPtr2DI1 
 module procedure ESMF_FieldEmptyCompLSPtr3DI1 
 module procedure ESMF_FieldEmptyCompLSPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompLSPtr5DI1 
 module procedure ESMF_FieldEmptyCompLSPtr6DI1 
 module procedure ESMF_FieldEmptyCompLSPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompLSPtr1DI2 
 module procedure ESMF_FieldEmptyCompLSPtr2DI2 
 module procedure ESMF_FieldEmptyCompLSPtr3DI2 
 module procedure ESMF_FieldEmptyCompLSPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompLSPtr5DI2 
 module procedure ESMF_FieldEmptyCompLSPtr6DI2 
 module procedure ESMF_FieldEmptyCompLSPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompLSPtr1DI4 
 module procedure ESMF_FieldEmptyCompLSPtr1DI8 
 module procedure ESMF_FieldEmptyCompLSPtr1DR4 
 module procedure ESMF_FieldEmptyCompLSPtr1DR8 
 module procedure ESMF_FieldEmptyCompLSPtr2DI4 
 module procedure ESMF_FieldEmptyCompLSPtr2DI8 
 module procedure ESMF_FieldEmptyCompLSPtr2DR4 
 module procedure ESMF_FieldEmptyCompLSPtr2DR8 
 module procedure ESMF_FieldEmptyCompLSPtr3DI4 
 module procedure ESMF_FieldEmptyCompLSPtr3DI8 
 module procedure ESMF_FieldEmptyCompLSPtr3DR4 
 module procedure ESMF_FieldEmptyCompLSPtr3DR8 
 module procedure ESMF_FieldEmptyCompLSPtr4DI4 
 module procedure ESMF_FieldEmptyCompLSPtr4DI8 
 module procedure ESMF_FieldEmptyCompLSPtr4DR4 
 module procedure ESMF_FieldEmptyCompLSPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompLSPtr5DI4 
 module procedure ESMF_FieldEmptyCompLSPtr5DI8 
 module procedure ESMF_FieldEmptyCompLSPtr5DR4 
 module procedure ESMF_FieldEmptyCompLSPtr5DR8 
 module procedure ESMF_FieldEmptyCompLSPtr6DI4 
 module procedure ESMF_FieldEmptyCompLSPtr6DI8 
 module procedure ESMF_FieldEmptyCompLSPtr6DR4 
 module procedure ESMF_FieldEmptyCompLSPtr6DR8 
 module procedure ESMF_FieldEmptyCompLSPtr7DI4 
 module procedure ESMF_FieldEmptyCompLSPtr7DI8 
 module procedure ESMF_FieldEmptyCompLSPtr7DR4 
 module procedure ESMF_FieldEmptyCompLSPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompMesh1DI1 
 module procedure ESMF_FieldEmptyCompMesh2DI1 
 module procedure ESMF_FieldEmptyCompMesh3DI1 
 module procedure ESMF_FieldEmptyCompMesh4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompMesh5DI1 
 module procedure ESMF_FieldEmptyCompMesh6DI1 
 module procedure ESMF_FieldEmptyCompMesh7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompMesh1DI2 
 module procedure ESMF_FieldEmptyCompMesh2DI2 
 module procedure ESMF_FieldEmptyCompMesh3DI2 
 module procedure ESMF_FieldEmptyCompMesh4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompMesh5DI2 
 module procedure ESMF_FieldEmptyCompMesh6DI2 
 module procedure ESMF_FieldEmptyCompMesh7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompMesh1DI4 
 module procedure ESMF_FieldEmptyCompMesh1DI8 
 module procedure ESMF_FieldEmptyCompMesh1DR4 
 module procedure ESMF_FieldEmptyCompMesh1DR8 
 module procedure ESMF_FieldEmptyCompMesh2DI4 
 module procedure ESMF_FieldEmptyCompMesh2DI8 
 module procedure ESMF_FieldEmptyCompMesh2DR4 
 module procedure ESMF_FieldEmptyCompMesh2DR8 
 module procedure ESMF_FieldEmptyCompMesh3DI4 
 module procedure ESMF_FieldEmptyCompMesh3DI8 
 module procedure ESMF_FieldEmptyCompMesh3DR4 
 module procedure ESMF_FieldEmptyCompMesh3DR8 
 module procedure ESMF_FieldEmptyCompMesh4DI4 
 module procedure ESMF_FieldEmptyCompMesh4DI8 
 module procedure ESMF_FieldEmptyCompMesh4DR4 
 module procedure ESMF_FieldEmptyCompMesh4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompMesh5DI4 
 module procedure ESMF_FieldEmptyCompMesh5DI8 
 module procedure ESMF_FieldEmptyCompMesh5DR4 
 module procedure ESMF_FieldEmptyCompMesh5DR8 
 module procedure ESMF_FieldEmptyCompMesh6DI4 
 module procedure ESMF_FieldEmptyCompMesh6DI8 
 module procedure ESMF_FieldEmptyCompMesh6DR4 
 module procedure ESMF_FieldEmptyCompMesh6DR8 
 module procedure ESMF_FieldEmptyCompMesh7DI4 
 module procedure ESMF_FieldEmptyCompMesh7DI8 
 module procedure ESMF_FieldEmptyCompMesh7DR4 
 module procedure ESMF_FieldEmptyCompMesh7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompMeshPtr1DI1 
 module procedure ESMF_FieldEmptyCompMeshPtr2DI1 
 module procedure ESMF_FieldEmptyCompMeshPtr3DI1 
 module procedure ESMF_FieldEmptyCompMeshPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompMeshPtr5DI1 
 module procedure ESMF_FieldEmptyCompMeshPtr6DI1 
 module procedure ESMF_FieldEmptyCompMeshPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompMeshPtr1DI2 
 module procedure ESMF_FieldEmptyCompMeshPtr2DI2 
 module procedure ESMF_FieldEmptyCompMeshPtr3DI2 
 module procedure ESMF_FieldEmptyCompMeshPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompMeshPtr5DI2 
 module procedure ESMF_FieldEmptyCompMeshPtr6DI2 
 module procedure ESMF_FieldEmptyCompMeshPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompMeshPtr1DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr1DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr1DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr1DR8 
 module procedure ESMF_FieldEmptyCompMeshPtr2DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr2DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr2DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr2DR8 
 module procedure ESMF_FieldEmptyCompMeshPtr3DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr3DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr3DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr3DR8 
 module procedure ESMF_FieldEmptyCompMeshPtr4DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr4DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr4DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompMeshPtr5DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr5DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr5DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr5DR8 
 module procedure ESMF_FieldEmptyCompMeshPtr6DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr6DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr6DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr6DR8 
 module procedure ESMF_FieldEmptyCompMeshPtr7DI4 
 module procedure ESMF_FieldEmptyCompMeshPtr7DI8 
 module procedure ESMF_FieldEmptyCompMeshPtr7DR4 
 module procedure ESMF_FieldEmptyCompMeshPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompXG1DI1 
 module procedure ESMF_FieldEmptyCompXG2DI1 
 module procedure ESMF_FieldEmptyCompXG3DI1 
 module procedure ESMF_FieldEmptyCompXG4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompXG5DI1 
 module procedure ESMF_FieldEmptyCompXG6DI1 
 module procedure ESMF_FieldEmptyCompXG7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompXG1DI2 
 module procedure ESMF_FieldEmptyCompXG2DI2 
 module procedure ESMF_FieldEmptyCompXG3DI2 
 module procedure ESMF_FieldEmptyCompXG4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompXG5DI2 
 module procedure ESMF_FieldEmptyCompXG6DI2 
 module procedure ESMF_FieldEmptyCompXG7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompXG1DI4 
 module procedure ESMF_FieldEmptyCompXG1DI8 
 module procedure ESMF_FieldEmptyCompXG1DR4 
 module procedure ESMF_FieldEmptyCompXG1DR8 
 module procedure ESMF_FieldEmptyCompXG2DI4 
 module procedure ESMF_FieldEmptyCompXG2DI8 
 module procedure ESMF_FieldEmptyCompXG2DR4 
 module procedure ESMF_FieldEmptyCompXG2DR8 
 module procedure ESMF_FieldEmptyCompXG3DI4 
 module procedure ESMF_FieldEmptyCompXG3DI8 
 module procedure ESMF_FieldEmptyCompXG3DR4 
 module procedure ESMF_FieldEmptyCompXG3DR8 
 module procedure ESMF_FieldEmptyCompXG4DI4 
 module procedure ESMF_FieldEmptyCompXG4DI8 
 module procedure ESMF_FieldEmptyCompXG4DR4 
 module procedure ESMF_FieldEmptyCompXG4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompXG5DI4 
 module procedure ESMF_FieldEmptyCompXG5DI8 
 module procedure ESMF_FieldEmptyCompXG5DR4 
 module procedure ESMF_FieldEmptyCompXG5DR8 
 module procedure ESMF_FieldEmptyCompXG6DI4 
 module procedure ESMF_FieldEmptyCompXG6DI8 
 module procedure ESMF_FieldEmptyCompXG6DR4 
 module procedure ESMF_FieldEmptyCompXG6DR8 
 module procedure ESMF_FieldEmptyCompXG7DI4 
 module procedure ESMF_FieldEmptyCompXG7DI8 
 module procedure ESMF_FieldEmptyCompXG7DR4 
 module procedure ESMF_FieldEmptyCompXG7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompXGPtr1DI1 
 module procedure ESMF_FieldEmptyCompXGPtr2DI1 
 module procedure ESMF_FieldEmptyCompXGPtr3DI1 
 module procedure ESMF_FieldEmptyCompXGPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompXGPtr5DI1 
 module procedure ESMF_FieldEmptyCompXGPtr6DI1 
 module procedure ESMF_FieldEmptyCompXGPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompXGPtr1DI2 
 module procedure ESMF_FieldEmptyCompXGPtr2DI2 
 module procedure ESMF_FieldEmptyCompXGPtr3DI2 
 module procedure ESMF_FieldEmptyCompXGPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompXGPtr5DI2 
 module procedure ESMF_FieldEmptyCompXGPtr6DI2 
 module procedure ESMF_FieldEmptyCompXGPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompXGPtr1DI4 
 module procedure ESMF_FieldEmptyCompXGPtr1DI8 
 module procedure ESMF_FieldEmptyCompXGPtr1DR4 
 module procedure ESMF_FieldEmptyCompXGPtr1DR8 
 module procedure ESMF_FieldEmptyCompXGPtr2DI4 
 module procedure ESMF_FieldEmptyCompXGPtr2DI8 
 module procedure ESMF_FieldEmptyCompXGPtr2DR4 
 module procedure ESMF_FieldEmptyCompXGPtr2DR8 
 module procedure ESMF_FieldEmptyCompXGPtr3DI4 
 module procedure ESMF_FieldEmptyCompXGPtr3DI8 
 module procedure ESMF_FieldEmptyCompXGPtr3DR4 
 module procedure ESMF_FieldEmptyCompXGPtr3DR8 
 module procedure ESMF_FieldEmptyCompXGPtr4DI4 
 module procedure ESMF_FieldEmptyCompXGPtr4DI8 
 module procedure ESMF_FieldEmptyCompXGPtr4DR4 
 module procedure ESMF_FieldEmptyCompXGPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompXGPtr5DI4 
 module procedure ESMF_FieldEmptyCompXGPtr5DI8 
 module procedure ESMF_FieldEmptyCompXGPtr5DR4 
 module procedure ESMF_FieldEmptyCompXGPtr5DR8 
 module procedure ESMF_FieldEmptyCompXGPtr6DI4 
 module procedure ESMF_FieldEmptyCompXGPtr6DI8 
 module procedure ESMF_FieldEmptyCompXGPtr6DR4 
 module procedure ESMF_FieldEmptyCompXGPtr6DR8 
 module procedure ESMF_FieldEmptyCompXGPtr7DI4 
 module procedure ESMF_FieldEmptyCompXGPtr7DI8 
 module procedure ESMF_FieldEmptyCompXGPtr7DR4 
 module procedure ESMF_FieldEmptyCompXGPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompGB1DI1 
 module procedure ESMF_FieldEmptyCompGB2DI1 
 module procedure ESMF_FieldEmptyCompGB3DI1 
 module procedure ESMF_FieldEmptyCompGB4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGB5DI1 
 module procedure ESMF_FieldEmptyCompGB6DI1 
 module procedure ESMF_FieldEmptyCompGB7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompGB1DI2 
 module procedure ESMF_FieldEmptyCompGB2DI2 
 module procedure ESMF_FieldEmptyCompGB3DI2 
 module procedure ESMF_FieldEmptyCompGB4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGB5DI2 
 module procedure ESMF_FieldEmptyCompGB6DI2 
 module procedure ESMF_FieldEmptyCompGB7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompGB1DI4 
 module procedure ESMF_FieldEmptyCompGB1DI8 
 module procedure ESMF_FieldEmptyCompGB1DR4 
 module procedure ESMF_FieldEmptyCompGB1DR8 
 module procedure ESMF_FieldEmptyCompGB2DI4 
 module procedure ESMF_FieldEmptyCompGB2DI8 
 module procedure ESMF_FieldEmptyCompGB2DR4 
 module procedure ESMF_FieldEmptyCompGB2DR8 
 module procedure ESMF_FieldEmptyCompGB3DI4 
 module procedure ESMF_FieldEmptyCompGB3DI8 
 module procedure ESMF_FieldEmptyCompGB3DR4 
 module procedure ESMF_FieldEmptyCompGB3DR8 
 module procedure ESMF_FieldEmptyCompGB4DI4 
 module procedure ESMF_FieldEmptyCompGB4DI8 
 module procedure ESMF_FieldEmptyCompGB4DR4 
 module procedure ESMF_FieldEmptyCompGB4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGB5DI4 
 module procedure ESMF_FieldEmptyCompGB5DI8 
 module procedure ESMF_FieldEmptyCompGB5DR4 
 module procedure ESMF_FieldEmptyCompGB5DR8 
 module procedure ESMF_FieldEmptyCompGB6DI4 
 module procedure ESMF_FieldEmptyCompGB6DI8 
 module procedure ESMF_FieldEmptyCompGB6DR4 
 module procedure ESMF_FieldEmptyCompGB6DR8 
 module procedure ESMF_FieldEmptyCompGB7DI4 
 module procedure ESMF_FieldEmptyCompGB7DI8 
 module procedure ESMF_FieldEmptyCompGB7DR4 
 module procedure ESMF_FieldEmptyCompGB7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldEmptyCompGBPtr1DI1 
 module procedure ESMF_FieldEmptyCompGBPtr2DI1 
 module procedure ESMF_FieldEmptyCompGBPtr3DI1 
 module procedure ESMF_FieldEmptyCompGBPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGBPtr5DI1 
 module procedure ESMF_FieldEmptyCompGBPtr6DI1 
 module procedure ESMF_FieldEmptyCompGBPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldEmptyCompGBPtr1DI2 
 module procedure ESMF_FieldEmptyCompGBPtr2DI2 
 module procedure ESMF_FieldEmptyCompGBPtr3DI2 
 module procedure ESMF_FieldEmptyCompGBPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGBPtr5DI2 
 module procedure ESMF_FieldEmptyCompGBPtr6DI2 
 module procedure ESMF_FieldEmptyCompGBPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldEmptyCompGBPtr1DI4 
 module procedure ESMF_FieldEmptyCompGBPtr1DI8 
 module procedure ESMF_FieldEmptyCompGBPtr1DR4 
 module procedure ESMF_FieldEmptyCompGBPtr1DR8 
 module procedure ESMF_FieldEmptyCompGBPtr2DI4 
 module procedure ESMF_FieldEmptyCompGBPtr2DI8 
 module procedure ESMF_FieldEmptyCompGBPtr2DR4 
 module procedure ESMF_FieldEmptyCompGBPtr2DR8 
 module procedure ESMF_FieldEmptyCompGBPtr3DI4 
 module procedure ESMF_FieldEmptyCompGBPtr3DI8 
 module procedure ESMF_FieldEmptyCompGBPtr3DR4 
 module procedure ESMF_FieldEmptyCompGBPtr3DR8 
 module procedure ESMF_FieldEmptyCompGBPtr4DI4 
 module procedure ESMF_FieldEmptyCompGBPtr4DI8 
 module procedure ESMF_FieldEmptyCompGBPtr4DR4 
 module procedure ESMF_FieldEmptyCompGBPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldEmptyCompGBPtr5DI4 
 module procedure ESMF_FieldEmptyCompGBPtr5DI8 
 module procedure ESMF_FieldEmptyCompGBPtr5DR4 
 module procedure ESMF_FieldEmptyCompGBPtr5DR8 
 module procedure ESMF_FieldEmptyCompGBPtr6DI4 
 module procedure ESMF_FieldEmptyCompGBPtr6DI8 
 module procedure ESMF_FieldEmptyCompGBPtr6DR4 
 module procedure ESMF_FieldEmptyCompGBPtr6DR8 
 module procedure ESMF_FieldEmptyCompGBPtr7DI4 
 module procedure ESMF_FieldEmptyCompGBPtr7DI8 
 module procedure ESMF_FieldEmptyCompGBPtr7DR4 
 module procedure ESMF_FieldEmptyCompGBPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


! !DESCRIPTION:
! This interface completes a {\tt ESMF\_Field} started with FieldEmptyCreate
! These methods all take {\tt ESMF\_Grid}, or {\tt ESMF\_LocStream}, or {\tt ESMF\_Mesh},
! or {\tt ESMF\_XGrid}, and Fortran data array or pointer as input to
! fill in the {\tt ESMF\_Field} internals.
  end interface
!EOPI

contains

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldEmptyCompAS"
!BOP
! !IROUTINE: ESMF_FieldEmptyComplete - Complete a Field from arrayspec

! !INTERFACE:
  ! Private name; call using ESMF_FieldEmptyComplete()
subroutine ESMF_FieldEmptyCompAS(field, arrayspec, keywordEnforcer, indexflag, &
 gridToFieldMap, ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, &
 pinflag, rc)

!
! !ARGUMENTS:
 type(ESMF_Field), intent(inout) :: field
 type(ESMF_ArraySpec), intent(in) :: arrayspec
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
 type(ESMF_Index_Flag), intent(in), optional :: indexflag
 integer, intent(in), optional :: gridToFieldMap(:)
 integer, intent(in), optional :: ungriddedLBound(:)
 integer, intent(in), optional :: ungriddedUBound(:)
 integer, intent(in), optional :: totalLWidth(:)
 integer, intent(in), optional :: totalUWidth(:)
 type(ESMF_Pin_Flag), intent(in), optional :: pinflag
 integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt pinflag} to provide access to DE sharing
! between PETs.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Complete an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array} based on arrayspec.
! The input {\tt ESMF\_Field} must have a status of
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}. After this call the completed {\tt ESMF\_Field}
! has a status of {\tt ESMF\_FIELDSTATUS\_COMPLETE}.
!
! The arguments are:
! \begin{description}
! \item[field]
! The input {\tt ESMF\_Field} with a status of
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}.
! \item[arrayspec]
! Data type and kind specification.
! \item[{[indexflag]}]
! Indicate how DE-local indices are defined. By default each DE's
! exclusive region is placed to start at the local index space origin,
! i.e. (1, 1, ..., 1). Alternatively the DE-local index space can be
! aligned with the global index space, if a global index space is well
! defined by the associated Grid. See section \ref{const:indexflag}
! for a list of valid indexflag options.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Grid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Grid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_Geom) :: geom
    type(ESMF_Index_Flag) :: g_indexflag
    type(ESMF_GridDecompType) :: decompType
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    type(ESMF_FieldStatus_Flag) :: status
    character(len=ESMF_MAXSTR) :: fieldName

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    call ESMF_FieldGet(field, status=status, name=fieldName, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if(status /= ESMF_FIELDSTATUS_GRIDSET) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
        msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    geom=field%ftypep%geom
    call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Here is where different handling of geomtype class branches off
    if(geomtype == ESMF_GEOMTYPE_GRID) then

      call ESMF_GeomGet(geom, grid=grid, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      call ESMF_GridGetDecompType(grid, decompType, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      if (decompType .eq. ESMF_GRID_ARBITRARY) then
        if ((present(indexflag) .or. present(totalLWidth)) .or. (present(totalUWidth))) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
                 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", &
                 ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
      endif

      call ESMF_GridGet(grid, indexFlag=g_indexflag, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return

      if (present(indexflag)) then
        if(.not. (g_indexflag .eq. indexflag)) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
               msg="- user specified indexflag must be identical with Grid indexflag", &
                 ESMF_CONTEXT, rcToReturn=rc)
          return
        endif
      endif

      ! Create Field from Geom
      call ESMF_FieldEmptyGBArraySpec(field, geom, &
          arrayspec, gridToFieldMap=gridToFieldMap, &
          ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, &
          totalLWidth=totalLWidth, totalUWidth=totalUWidth, &
          pinflag=pinflag, rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    endif ! Grid

    if(geomtype == ESMF_GEOMTYPE_MESH .or. &
       geomtype == ESMF_GEOMTYPE_LOCSTREAM .or. &
       geomtype == ESMF_GEOMTYPE_XGRID) then

      if ((present(indexflag) .or. present(totalLWidth)) .or. (present(totalUWidth))) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- indexflag, totalLWidth or totalUWidth are not allowed for Mesh, LS, XGrid", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif

      call ESMF_FieldConstructIANew(field%ftypep, geom, arrayspec, &
                            gridToFieldMap=gridToFieldMap, &
                            ungriddedLBound=ungriddedLBound, &
                            ungriddedUBound=ungriddedUBound, &
                            hasBase=.true., name=fieldName, &
                            pinflag=pinflag, rc=localrc)

     if (ESMF_LogFoundError(localrc, &
       ESMF_ERR_PASSTHRU, &
       ESMF_CONTEXT, rcToReturn=rc)) return

    endif

    if(present(rc)) rc = ESMF_SUCCESS

end subroutine ESMF_FieldEmptyCompAS
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldEmptyCompTK"
!BOP
! !IROUTINE: ESMF_FieldEmptyComplete - Complete a Field from typekind

! !INTERFACE:
  ! Private name; call using ESMF_FieldEmptyComplete()
subroutine ESMF_FieldEmptyCompTK(field, typekind, keywordEnforcer, indexflag, &
 gridToFieldMap, ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, &
 pinflag, rc)

!
! !ARGUMENTS:
 type(ESMF_Field), intent(inout) :: field
 type(ESMF_TypeKind_Flag), intent(in) :: typekind
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
 type(ESMF_Index_Flag), intent(in), optional :: indexflag
 integer, intent(in), optional :: gridToFieldMap(:)
 integer, intent(in), optional :: ungriddedLBound(:)
 integer, intent(in), optional :: ungriddedUBound(:)
 integer, intent(in), optional :: totalLWidth(:)
 integer, intent(in), optional :: totalUWidth(:)
 type(ESMF_Pin_Flag), intent(in), optional :: pinflag
 integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[8.1.0] Added argument {\tt pinflag} to provide access to DE sharing
! between PETs.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! \begin{sloppypar}
! Complete an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array} based on typekind.
! The input {\tt ESMF\_Field} must have a status of
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}. After this call the completed {\tt ESMF\_Field}
! has a status of {\tt ESMF\_FIELDSTATUS\_COMPLETE}.
! \end{sloppypar}
!
! For an example and
! associated documentation using this method see section
! \ref{sec:field:usage:partial_creation}.
!
!
! The arguments are:
! \begin{description}
! \item[field]
! \begin{sloppypar}
! The input {\tt ESMF\_Field} with a status of
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}.
! \end{sloppypar}
! \item[typekind]
! Data type and kind specification.
! \item[{[indexflag]}]
! Indicate how DE-local indices are defined. By default each DE's
! exclusive region is placed to start at the local index space origin,
! i.e. (1, 1, ..., 1). Alternatively the DE-local index space can be
! aligned with the global index space, if a global index space is well
! defined by the associated Grid. See section \ref{const:indexflag}
! for a list of valid indexflag options.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! If the Field dimCount is less than the Grid dimCount then the default
! gridToFieldMap will contain zeros for the rightmost entries. A zero
! entry in the {\tt gridToFieldMap} indicates that the particular
! Grid dimension will be replicating the Field across the DEs along
! this direction.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc ! Local error code
    type(ESMF_ArraySpec) :: arrayspec
    integer :: rank, ungridded_dimcount, ungridded_ldimcount, ungridded_udimcount
    integer :: i, memDimCount, grid_repdimcount
    type(ESMF_Geom) :: geom
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    type(ESMF_FieldStatus_Flag) :: status
    type(ESMF_GridDecompType) :: decompType

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL

    ! Initialize return code
    if(present(rc)) rc = ESMF_RC_NOT_IMPL

    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc)

    call ESMF_FieldGet(field, status=status, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if(status /= ESMF_FIELDSTATUS_GRIDSET) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
        msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    ! compute arrayspec rank, distgrid dimcount is 1 for Mesh, LocStream and XGrid
    ! arrayspec rank = distgrid_dimcount + ungridded_dimcount
    ungridded_dimcount = 0
    if(present(ungriddedLBound)) then
      ungridded_ldimcount = size(ungriddedLBound)
      ungridded_dimcount = ungridded_ldimcount
    endif
    if(present(ungriddedUBound)) then
      ungridded_udimcount = size(ungriddedUBound)
      ungridded_dimcount = ungridded_udimcount
    endif

    if(present(ungriddedLBound) .and. present(ungriddedUBound)) then
      if(ungridded_ldimcount /= ungridded_udimcount) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, &
          msg="- ungriddedLBound size must equal to ungriddedUBound size", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
   endif

    ! Get geom
    geom=field%ftypep%geom

    ! Get number of replicated dims
    grid_repdimcount = 0
    if (present(gridToFieldMap)) then
       do i = 1, size(gridToFieldMap)
           if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
       enddo
    endif

    ! Get rank of geom
    call ESMF_GeomGet(geom, rank=memDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    ! Calculate rank from dimension information
    rank = ungridded_dimcount + memDimCount - grid_repdimcount

    ! Set ArraySpec from rank and typekind input
    call ESMF_ArraySpecSet(arrayspec, rank=rank, typekind=typekind, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Complete Field
    call ESMF_FieldEmptyComplete(field, arrayspec, &
      indexflag=indexflag, gridToFieldMap=gridToFieldMap, &
      ungriddedLBound=ungriddedLBound, ungriddedUBound=ungriddedUBound, &
      totalLWidth=totalLWidth, totalUWidth=totalUWidth, &
      pinflag=pinflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Return successfully
    if (present(rc)) rc = ESMF_SUCCESS

end subroutine ESMF_FieldEmptyCompTK
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldEmptyGBArraySpec"
!BOPI
! !IROUTINE: ESMF_FieldEmptyGBArraySpec - Complete a Field from Geom and ArraySpec.

! !INTERFACE:
  subroutine ESMF_FieldEmptyGBArraySpec(field, geom, arrayspec, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, pinflag, rc)
!
! !ARGUMENTS:
    type(ESMF_Field), intent(inout) :: field
    type(ESMF_Geom), intent(in) :: geom
    type(ESMF_ArraySpec), intent(in) :: arrayspec
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    type(ESMF_Pin_Flag), intent(in), optional :: pinflag
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Complete an {\tt ESMF\_Field} and allocate space internally for an
! {\tt ESMF\_Array}.
! The arguments are:
! \begin{description}
! \item [field]
! {\tt ESMF\_Field} object.
! \item [geom]
! {\tt ESMF\_Geom} object.
! \item [arrayspec]
! Data type and kind specification.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOPI
!------------------------------------------------------------------------------
    type(ESMF_FieldType), pointer :: ftype ! Pointer to new field
    integer :: localrc ! Local error code
    logical :: rcpresent ! Return code present
    integer :: memDimCount, fieldDimCount, fieldUngriddedDimCount
    integer :: gridDimCount, gridDimCount_norep, grid_repdimcount
    integer :: fieldUndistDimCount
    integer :: i,j
    integer :: localGridToFieldMap(ESMF_MAXDIM)
    integer :: localMaxHaloLWidth (ESMF_MAXDIM)
    integer :: localMaxHaloUWidth (ESMF_MAXDIM)
    integer :: distgridToGridMap(ESMF_MAXDIM)
    logical :: flipflop(ESMF_MAXDIM)
    logical :: found
    type(ESMF_GridDecompType) :: decompType
    type(ESMF_GeomType_Flag) :: geomtype
    type(ESMF_Grid) :: grid
    integer :: arbdim
    character(len=ESMF_MAXSTR) :: fieldName

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL
    rcpresent = .FALSE.

    ! Initialize return code
    if(present(rc)) then
      rcpresent=.TRUE.
      rc = ESMF_RC_NOT_IMPL
    endif

    ! check init status of input parameters
    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc)
    ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc)
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    call ESMF_FieldGet(field, name=fieldName, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Check if geom is a grid, if so, check if it is arbitrary
    decompType = ESMF_GRID_NONARBITRARY
    call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    if (geomtype .eq. ESMF_GEOMTYPE_GRID) then
       call ESMF_GeomGet(geom, grid=grid, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
       call ESMF_GridGetDecompType(grid, decompType, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
           ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Get rank of proposed Field.
    call ESMF_ArraySpecGet(arrayspec, rank=memDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    ! Get number of grid dimensions, number
    ! of distributed grid dimensions, distgrid,
    ! number of ungridded Field dimensions,
    ! and number of undistributed Field Dimensions
    call ESMF_GeomGet(geom, dimCount=gridDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

    ! Error Check Input
    if (present(gridToFieldMap)) then
        if (size(gridToFieldMap) .ne. gridDimCount) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
             msg="- gridToFieldMap size must equal to grid_rank", &
             ESMF_CONTEXT, rcToReturn=rc)
            return
        endif
    endif

    grid_repdimcount = 0
    localGridToFieldMap = -1
    if (present(gridToFieldMap)) then
       localGridToFieldMap(1:gridDimCount) = &
         gridToFieldMap (1:gridDimCount)
       do i = 1, size(gridToFieldMap)
           if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
       enddo
    else
      do i = 1, gridDimCount
        localGridToFieldMap(i) = i
      enddo
    endif
    gridDimCount_norep = gridDimCount - grid_repdimcount

    if (decompType .eq. ESMF_GRID_NONARBITRARY) then
            fieldDimCount = memDimCount
    else
        call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, &
            rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return
        ! find out how many grid dimensions are arbitrarily distributed and calculate
        ! fieldDimCount accordingly
        arbdim = 0
        do i=1,gridDimCount
          if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1
        enddo
        fieldDimCount = memDimCount + arbdim - 1

        ! If there is any replicated dimension, check if any of the arb. dimensions are replicated.
        ! If one arb dimension is replicated, all the arb. dimensions have to be replicated
        if (grid_repdimcount .ne. 0) then
          do i = 1,gridDimCount
            if(localGridToFieldMap(i) == 0) then
                found = .false.
                do j=1,arbdim
                    if (distgridToGridMap(j) .eq. i) found = .true.
                enddo
                if (found) then
                  ! one arb.dimension is a replicated dimension, check if other arb dimensions are
                  ! also replicated
                  do j=1,arbdim
                    if (distgridToGridMap(j) .ne. i) then
                        if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then
                            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
  msg="- Arb. grid dimensions have to be either all replicated or not replicated", &
  ESMF_CONTEXT, rcToReturn=rc)
                            return
                        endif
                    endif
                  enddo
                  ! all arb. dimension are replication, jump out of the first do loop
                  ! fieldDimCount should be the same as the memDimCount
                  fieldDimCount = memDimCount
                  exit
                endif
             endif
           enddo
        endif
    endif

    if(fieldDimCount .lt. gridDimCount_norep) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
        msg="- Field rank must be greater than or equal to its gridded rank", &
        ESMF_CONTEXT, rcToReturn=rc)
      return
    endif

    if(fieldDimCount .gt. gridDimCount_norep) then
      if( (.not. present(ungriddedLBound)) .or. &
          (.not. present(ungriddedUBound)) ) then
        call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
          msg="- ungridded bounds must be present when Field has ungridded dimension(s)", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
    endif

    fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount
    fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount

    if (present(ungriddedLBound)) then
       if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
               msg="- ungriddedLBound size must equal to array_rank-grid_rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    if (present(ungriddedUBound)) then
       if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
                 msg="- ungriddedUBound size must equal to array_rank-grid_rank", &
                 ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    if (present(totalLWidth)) then
        if (size(totalLWidth) .ne. gridDimCount_norep) then
            call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
              msg="- totalLWidth must equal to grid dimCount", &
                ESMF_CONTEXT, rcToReturn=rc)
            return
        endif
    endif

    if (present(totalUWidth)) then
       if (size(totalUWidth) .ne. gridDimCount_norep) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, &
              msg="- totalUWidth must equal to grid dimCount", &
                ESMF_CONTEXT, rcToReturn=rc)
           return
       endif
    endif

    ! gridToFieldMap elements must be in range 1...fieldRank and unique
    ! algorithm to check element uniqueness:
    ! run time: O(ESMF_MAXDIM)
    ! memory: O(2*ESMF_MAXDIM)
    ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector
    flipflop = .false.
    do i = 1, gridDimCount
       if(localGridToFieldMap(i) .lt. 0 .and. &
         localGridToFieldMap(i) .gt. fieldDimCount) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
             msg="- gridToFieldMap element must be within range 0...array rank", &
             ESMF_CONTEXT, rcToReturn=rc)
           return
       endif
       if(localGridToFieldMap(i) /= 0) then
           if(flipflop(localGridToFieldMap(i))) then
             call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="- gridToFieldMap element must be unique", &
               ESMF_CONTEXT, rcToReturn=rc)
             return
           endif
           flipflop(localGridToFieldMap(i)) = .true.
       endif
    enddo

     if(present(totalLWidth)) then
       localMaxHaloLWidth(1:gridDimCount_norep) = &
          totalLWidth (1:gridDimCount_norep)
     else
          localMaxHaloLWidth = 0
     endif

     if(present(totalUWidth)) then
       localMaxHaloUWidth(1:gridDimCount_norep) = &
          totalUWidth (1:gridDimCount_norep)
     else
          localMaxHaloUWidth = 0
     endif

    ! assign ftype
    ftype => field%ftypep

    ! Call construction method to allocate and initialize field internals.
    if (decompType .eq. ESMF_GRID_NONARBITRARY) then
       call ESMF_FieldConstructIANew(ftype, geom, arrayspec, &
                                localGridToFieldMap(1:gridDimCount), ungriddedLBound, &
                                ungriddedUBound, localMaxHaloLWidth(1:gridDimCount_norep), &
                                localMaxHaloUWidth(1:gridDimCount_norep), &
                                hasBase=.true., name=fieldName, &
                                pinflag=pinflag, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return
    else
       call ESMF_FieldConstructIANew(ftype, geom, arrayspec, &
                              localGridToFieldMap(1:gridDimCount), ungriddedLBound, &
                              ungriddedUBound, &
                              hasBase=.true., name=fieldName, &
                              pinflag=pinflag, rc=localrc)
       if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    call ESMF_FieldValidate(field, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) then
      call ESMF_BaseSetStatus(ftype%base, ESMF_STATUS_INVALID, rc=localrc)
      return
    endif

    if(rcpresent) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldEmptyGBArraySpec

!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldConstructIANew"
!BOPI
! !IROUTINE: ESMF_FieldConstructIANew - Construct the internals of a Field.

! !INTERFACE:
  subroutine ESMF_FieldConstructIANew(ftype, geom, arrayspec, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, hasBase, pinflag, name, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldType), pointer :: ftype
    type(ESMF_Geom) :: geom
    type(ESMF_ArraySpec), intent(in) :: arrayspec
    integer, intent(in), optional :: gridToFieldMap(:)
    integer, intent(in), optional :: ungriddedLBound(:)
    integer, intent(in), optional :: ungriddedUBound(:)
    integer, intent(in), optional :: totalLWidth(:)
    integer, intent(in), optional :: totalUWidth(:)
    logical, intent(in), optional :: hasBase
    type(ESMF_Pin_Flag), intent(in), optional:: pinflag
    character (len=*), intent(in), optional:: name
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Constructs all {\tt ESMF\_Field} internals, including the allocation
! of a data {\tt ESMF\_Array}. TODO: this is missing a counts argument,
! which is required if the arrayspec rank is greater than the {\tt grid} rank.
! Either that, or we must enforce that a datamap comes in, and it
! contains the counts for non-grid dims.
!
! The arguments are:
! \begin{description}
! \item [ftype]
! Pointer to an {\tt ESMF\_Field} object.
! \item [geom]
! {\tt ESMF\_Geom} object.
! \item [arrayspec]
! Data specification.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of gridded dimensions in the Field. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[hasBase]}]
! Input ftype already has a base associated with it.
! \item[{[pinflag]}]
! Specify which type of resource DEs are pinned to. See section
! \ref{const:pin_flag} for a list of valid pinning options.
! The default is to pin DEs to PETs, i.e. only the PET on which a DE
! was created considers the DE as local.
! \item [{[name]}]
! Field name.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
!
! \end{description}
!EOPI
!------------------------------------------------------------------------------
    integer :: localrc
    type(ESMF_Array) :: array
    integer :: i, arrayRank, gridDimCount, gridDimCount_norep, grid_repdimcount
    type(ESMF_DistGrid) :: distgrid
    type(ESMF_Index_Flag) :: indexflag
    integer, pointer :: distgridToArrayMap(:)
    integer, pointer :: arrayLBound(:),arrayUBound(:)
    integer :: ungriddedDimCount
    integer :: distgridDimCount, distgridDimCount_norep
    logical :: l_hasBase

    ! Initialize return code
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Check init status of arguments
    ESMF_INIT_CHECK_SHALLOW(ESMF_ArraySpecGetInit, arrayspec, rc)

    l_hasBase = .false.
    if(present(hasBase)) l_hasBase = hasBase

    if(.not. l_hasBase) then
        call ESMF_BaseCreate(ftype%base, "Field", name, 0, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    call ESMF_ArraySpecGet(arrayspec, rank=arrayRank, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return


    call ESMF_GeomGet(geom, distgrid=distgrid, dimCount=gridDimCount, &
                 indexflag=indexflag, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    grid_repdimcount = 0
    if (present(gridToFieldMap)) then
       do i = 1, size(gridToFieldMap)
           if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
       enddo
    endif
    gridDimCount_norep = gridDimCount - grid_repdimcount

    ! Get the ungridded dimCount
    ungriddedDimCount=0
    if (present(ungriddedUBound)) then
       ungriddedDimCount=size(ungriddedUBound)
    endif

    call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return
    ! allocate distgridToArrayMap
    allocate(distgridToArrayMap(distgridDimCount) , stat=localrc)
    if (ESMF_LogFoundAllocError(localrc, msg="Allocating distgridToArrayMap", &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! allocate undistributed Bounds
    if(ungriddedDimCount /= 0) then
      allocate(arrayLBound(ungriddedDimCount) , stat=localrc)
      if (ESMF_LogFoundAllocError(localrc, msg="Allocating gridLBound", &
        ESMF_CONTEXT, rcToReturn=rc)) return
      allocate(arrayUBound(ungriddedDimCount) , stat=localrc)
      if (ESMF_LogFoundAllocError(localrc, msg="Allocating gridUBound", &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! Get dimmap and undistibuted bounds
      call ESMF_GeomGetArrayInfo(geom, &
                              gridToFieldMap=gridToFieldMap, &
                              ungriddedLBound=ungriddedLBound, &
                              ungriddedUBound=ungriddedUBound, &
                              distgridToArrayMap=distgridToArrayMap, &
                              undistLBound=arrayLBound, &
                              undistUBound=arrayUBound, &
                              rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! create Array
      ! For arbitrary grid: distgridDim is different from gridDimCount and
      ! the dimension of the computationalEdgeLWidth, computationslEdgeUWidth
      ! are the dimension of the array,
      distgridDimCount_norep = arrayRank - ungriddedDimCount
      array=ESMF_ArrayCreate(distgrid=distgrid, arrayspec=arrayspec, &
                pinflag=pinflag, distgridToArrayMap=distgridToArrayMap, &
                totalLWidth=totalLWidth, totalUWidth=totalUWidth, &
                indexflag=indexflag, &
                undistLBound=arrayLBound, undistUBound=arrayUBound, &
                name=name, &
                rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    else
      nullify (arrayLBound)
      nullify (arrayUBound)
      ! Get dimmap and undistibuted bounds
      call ESMF_GeomGetArrayInfo(geom, &
                              gridToFieldMap=gridToFieldMap, &
                              ungriddedLBound=ungriddedLBound, &
                              ungriddedUBound=ungriddedUBound, &
                              distgridToArrayMap=distgridToArrayMap, &
                              rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return

      ! create Array
      ! For arbitrary grid: distgridDim is different from gridDimCount and
      ! the dimension of the computationalEdgeLWidth, computationslEdgeUWidth
      ! are the dimension of the array,
      distgridDimCount_norep = arrayRank - ungriddedDimCount
      array=ESMF_ArrayCreate(distgrid=distgrid, arrayspec=arrayspec, &
                pinflag=pinflag, distgridToArrayMap=distgridToArrayMap, &
                totalLWidth=totalLWidth, totalUWidth=totalUWidth, &
                indexflag=indexflag, &
                name=name, &
                rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! Default of gridToFieldMap should be {1,2,3...}
    if (.not. present(gridToFieldMap)) then
        do i = 1, ESMF_MAXDIM
          ftype%gridToFieldMap(i) = i
        enddo
    else
       ftype%gridToFieldMap(1:size(gridToFieldMap)) = gridToFieldMap
    endif

    if(present(ungriddedLBound)) &
       ftype%ungriddedLBound(1:size(ungriddedLBound)) = ungriddedLBound
    if(present(ungriddedUBound)) &
       ftype%ungriddedUBound(1:size(ungriddedUBound)) = ungriddedUBound
    if(present(totalLWidth)) &
       ftype%totalLWidth(1:size(totalLWidth)) = totalLWidth
    if(present(totalUWidth)) &
       ftype%totalUWidth(1:size(totalUWidth)) = totalUWidth

    ftype%array = array
    ftype%array_internal = .true.
    ftype%geom = geom
    ftype%status = ESMF_FIELDSTATUS_COMPLETE
    ftype%dimCount = gridDimCount_norep + ungriddedDimCount
    if(.not. l_hasBase) then
        call ESMF_BaseSetStatus(ftype%base, ESMF_STATUS_READY, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    ! cleanup
    deallocate(distgridToArrayMap)
    if (ungriddedDimCount /= 0) then
      deallocate(arrayLBound)
      deallocate(arrayUBound)
    end if

    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldConstructIANew
!------------------------------------------------------------------------------


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldConstructIANewArray"
!BOPI
! !IROUTINE: ESMF_FieldConstructIANewArray - Construct the internals of a Field.

! !INTERFACE:
  subroutine ESMF_FieldConstructIANewArray(ftype, geom, array, datacopyflag, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, name, vm, rc)
!
! !ARGUMENTS:
    type(ESMF_FieldType), pointer :: ftype
    type(ESMF_Geom) :: geom
    type(ESMF_Array), intent(in) :: array
    type(ESMF_DataCopy_Flag), intent(in) :: datacopyflag
    integer, intent(in) :: gridToFieldMap(:)
    integer, intent(in),optional :: ungriddedLBound(:)
    integer, intent(in),optional :: ungriddedUBound(:)
    integer, intent(in),optional :: totalLWidth(:)
    integer, intent(in),optional :: totalUWidth(:)
    character (len=*), intent(in), optional :: name
    type(ESMF_VM), intent(in), optional :: vm
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Constructs all {\tt ESMF\_Field} internals, including the allocation
! of a data {\tt ESMF\_Array}.
!
! The arguments are:
! \begin{description}
! \item [ftype]
! Pointer to an {\tt ESMF\_Field} object.
! \item [geom]
! {\tt ESMF\_Geom} object.
! \item [array]
! Data.
! \item [datacopyflag]
! Whether to copy the existing data space or reference directly. Valid
! values are {\tt ESMF\_DATACOPY\_VALUE} or {\tt ESMF\_DATACOPY\_REFERENCE} (default).
! \item [staggerloc]
! Stagger location of data in grid cells. For valid
! predefined values see section \ref{const:staggerloc}.
! To create a custom stagger location see section
! \ref{sec:usage:staggerloc:adv}. The default
! value is {\tt ESMF\_STAGGERLOC\_CENTER}.
! \item [gridToFieldMap]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The values of all {\tt gridToFieldMap} entries must be greater than or equal
! to one and smaller than or equal to the {\tt field} rank.
! It is erroneous to specify the same {\tt gridToFieldMap} entry
! multiple times. The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. When field dimension count is
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound
! must be specified. When both are specified the values are checked
! for consistency. Note that the the ordering of
! these ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of this array is the number
! of dimensions in the {\tt grid}. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of this array is the number
! of dimensions in the {\tt grid}. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! \item [{[name]}]
! Field name.
! \item[{[vm]}]
! If present, the Field object is constructed on the specified
! {\tt ESMF\_VM} object. The default is to construct on the VM of the
! current component context.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOPI
!------------------------------------------------------------------------------
    integer :: localrc
    type(ESMF_Array) :: newarray
    integer :: ungriddedDimCount, gridDimCount, gridDimCount_norep
    integer :: grid_repdimcount, i, dimCount, lDeCount, j
    integer, allocatable :: tlw(:,:), tuw(:,:)

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! Construct a default name if one is not given
    call ESMF_BaseCreate(ftype%base, "Field", name, 0, vm=vm, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! make sure the array is a valid object first.
    call ESMF_ArrayValidate(array, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    call ESMF_GeomGet(geom, dimCount=gridDimCount, &
                      rc=localrc)
    if (ESMF_LogFoundError(localrc, &
       ESMF_ERR_PASSTHRU, &
       ESMF_CONTEXT, rcToReturn=rc)) return

    grid_repdimcount = 0
    do i = 1, size(gridToFieldMap)
        if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1
    enddo
    gridDimCount_norep = gridDimCount - grid_repdimcount

    ! Get the ungridded dimCount
    ungriddedDimCount = 0
    if(present(ungriddedUBound)) then
        ungriddedDimCount=size(ungriddedUBound)
    endif

    if(present(ungriddedLBound)) then
        ftype%ungriddedLBound(1:size(ungriddedLBound)) = ungriddedLBound
    else
        ftype%ungriddedLBound = -1
    endif
    if(present(ungriddedUBound)) then
        ftype%ungriddedUBound(1:size(ungriddedUBound)) = ungriddedUBound
    else
        ftype%ungriddedUBound = -1
    endif

    ! make sure the user passed in haloWidths are consistent with those stored in Array
    call ESMF_ArrayGet(array, dimCount=dimCount, localDeCount=lDeCount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return
!gjt: it is okay to have no localDEs, remove the commented code eventually!
! if(lDeCount .lt. 1) then
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
! msg="- the number of local DEs in array is less than 1 ", &
! ESMF_CONTEXT, rcToReturn=rc)
! return
! endif
    allocate(tlw(dimCount, lDeCount), tuw(dimCount, lDeCount))
    call ESMF_ArrayGet(array, totalLWidth=tlw, totalUWidth=tuw, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return

    ! Field only support single DE halowidth spec, so the width in Array must be the same
    ! for all ldes
    do i = 2, lDeCount
      do j = 1, dimCount
        if(tlw(j, i) /= tlw(j,1) .or. tuw(j,i) /= tuw(j,1)) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
          msg="- totalWidth in Array varies with localDE, not supported by Field", &
          ESMF_CONTEXT, rcToReturn=rc)
          return
        endif
      enddo
    enddo

    if(present(totalLWidth)) then
        ftype%totalLWidth(1:gridDimCount_norep) = &
         totalLWidth (1:gridDimCount_norep)
    else
         ftype%totalLWidth = 0
    endif

    if(present(totalUWidth)) then
        ftype%totalUWidth(1:gridDimCount_norep) = &
         totalUWidth (1:gridDimCount_norep)
    else
         ftype%totalUWidth = 0
    endif
    ftype%gridToFieldMap(1:size(gridToFieldMap)) = gridToFieldMap

    ! default datacopyflag value is ESMF_DATACOPY_REFERENCE
    ftype%array_internal = .false.
    if(datacopyflag == ESMF_DATACOPY_REFERENCE) then
        ftype%array = array
    else
        newarray = ESMF_ArrayCreate(array, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        ftype%array = newarray
        ftype%array_internal = .true.
    endif

    ftype%geom = geom
    ftype%status = ESMF_FIELDSTATUS_COMPLETE
    call ESMF_BaseSetStatus(ftype%base, ESMF_STATUS_READY, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
      ESMF_ERR_PASSTHRU, &
      ESMF_CONTEXT, rcToReturn=rc)) return
    ftype%dimCount = gridDimCount_norep + ungriddedDimCount

    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldConstructIANewArray
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldEmptyComplete - Complete a Field from Fortran array 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldEmptyComplete() 
! subroutine ESMF_FieldEmptyComp<rank><type><kind>(field, & 
! farray, indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
! ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 
! 
! !ARGUMENTS: 
! type(ESMF_Field), intent(inout) :: field 
! <type> (ESMF_KIND_<kind>),intent(in), target :: farray(<rank>) 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! integer, intent(out), optional :: rc 
! 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! \begin{sloppypar} 
! Complete an {\tt ESMF\_Field} and allocate space internally for an 
! {\tt ESMF\_Array} based on typekind. 
! The input {\tt ESMF\_Field} must have a status of 
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}. After this call the completed {\tt ESMF\_Field} 
! has a status of {\tt ESMF\_FIELDSTATUS\_COMPLETE}. 
! \end{sloppypar} 
! 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! 
! For an example and 
! associated documentation using this method see section 
! \ref{sec:field:usage:create_empty}. 
! 
! 
! The arguments are: 
! \begin{description} 
! \item [field] 
! The input {\tt ESMF\_Field} with a status of 
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}. 
! The {\tt ESMF\_Field} will have the same dimension 
! (dimCount) as the rank of the {\tt farray}. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the {\tt field}. 
! The {\tt field} dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item [indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item [{[datacopyflag]}] 
! Indicates whether to copy the {\tt farray} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The 
! default is to map all of the {\tt grid}'s dimensions against the 
! lowest dimensions of the {\tt farray} in sequence, i.e. 
! {\tt gridToFieldMap} = (/1,2,3,.../). 
! Unmapped {\tt farray} dimensions are undistributed Field 
! dimensions. 
! All {\tt gridToFieldMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Field dimCount. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Field dimCount is less than the Grid dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Grid dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt field}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt field}. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the {\tt field}. However, ordering of the elements 
! needs to be the same as they appear in the {\tt field}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt field}. That is, for each gridded dimension the 
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the {\tt field}. However, ordering of the elements 
! needs to be the same as they appear in the {\tt field}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt field}. That is, for each gridded dimension the 
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp1Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp2Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp3Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp4Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp5Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp6Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp7Di1(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp1Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp2Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp3Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp4Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp5Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp6Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp7Di2(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp1Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp1Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp1Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp1Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp2Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp2Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp2Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp2Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp3Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp3Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp3Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp3Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp4Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp4Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp4Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp4Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp5Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp5Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp5Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp5Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp6Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp6Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp6Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp6Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp7Di4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp7Di8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp7Dr4(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp7Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyComp" 
 subroutine ESMF_FieldEmptyComp7Dr8(field, farray, & 
 indexflag, keywordEnforcer, datacopyflag, gridToFieldMap, & 
 ungriddedLBound, ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag):: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyComp7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldEmptyComplete - Complete a Field from Fortran array pointer 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldEmptyComplete() 
! subroutine ESMF_FieldEmptyCompPtr<rank><type><kind>(field, & 
! farrayPtr, keywordEnforcer, datacopyflag, gridToFieldMap, & 
! totalLWidth, totalUWidth, rc) 
! 
! !ARGUMENTS: 
! type(ESMF_Field), intent(inout) :: field 
! <type> (ESMF_KIND_<kind>), pointer :: farrayPtr(<rank>) 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! integer, intent(out), optional :: rc 
! 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! \begin{sloppypar} 
! Complete an {\tt ESMF\_Field} and allocate space internally for an 
! {\tt ESMF\_Array} based on typekind. 
! The input {\tt ESMF\_Field} must have a status of 
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}. After this call the completed {\tt ESMF\_Field} 
! has a status of {\tt ESMF\_FIELDSTATUS\_COMPLETE}. 
! 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried and deallocated when 
! datacopyflag is {\tt ESMF\_DATACOPY\_REFERENCE}. Note that the {\tt ESMF\_FieldDestroy} call does not deallocate 
! the Fortran data pointer in this case. This gives user more flexibility over memory management. 
! \end{sloppypar} 
! 
! The arguments are: 
! \begin{description} 
! \item [field] 
! The input {\tt ESMF\_Field} with a status of 
! {\tt ESMF\_FIELDSTATUS\_GRIDSET}. 
! The {\tt ESMF\_Field} will have the same dimension 
! (dimCount) as the rank of the {\tt farrayPtr}. 
! \item [farrayPtr] 
! Native Fortran data pointer to be copied/referenced in the {\tt field}. 
! The {\tt field} dimension (dimCount) will be the same as the dimCount 
! for the {\tt farrayPtr}. 
! \item [{[datacopyflag]}] 
! Indicates whether to copy the {\tt farrayPtr} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt grid}'s dimCount. The list elements map each dimension 
! of the {\tt grid} to a dimension in the {\tt farrayPtr} by 
! specifying the appropriate {\tt farrayPtr} dimension index. The 
! default is to map all of the {\tt grid}'s dimensions against the 
! lowest dimensions of the {\tt farrayPtr} in sequence, i.e. 
! {\tt gridToFieldMap} = (/1,2,3,.../). 
! Unmapped {\tt farrayPtr} dimensions are undistributed Field 
! dimensions. 
! All {\tt gridToFieldMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Field dimCount. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Field dimCount is less than the Grid dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Grid dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the {\tt field}. However, ordering of the elements 
! needs to be the same as they appear in the {\tt field}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt field}. That is, for each gridded dimension the 
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the {\tt field}. However, ordering of the elements 
! needs to be the same as they appear in the {\tt field}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt field}. That is, for each gridded dimension the 
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr1Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr2Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr3Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr4Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr5Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr6Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr7Di1(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr1Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr2Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr3Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr4Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr5Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr6Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr7Di2(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr1Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr1Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr1Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr1Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr2Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr2Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr2Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr2Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr3Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr3Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr3Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr3Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr4Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr4Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr4Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr4Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr5Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr5Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr5Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr5Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr6Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr6Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr6Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr6Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr7Di4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr7Di8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr7Dr4(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr7Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompPtr" 
 subroutine ESMF_FieldEmptyCompPtr7Dr8(field, farrayPtr, & 
 keywordEnforcer, datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field) :: field 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer :: localrc 
 type(ESMF_Geom) :: geom 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 type(ESMF_FieldStatus_Flag) :: status 
 type(ESMF_GridDecompType) :: decompType 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_FieldGet(field, status=status, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(status /= ESMF_FIELDSTATUS_GRIDSET) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- can only use ESMF_FieldEmptyComplete with Fields having geom set", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 geom=field%ftypep%geom 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(geomtype == ESMF_GEOMTYPE_GRID) then 

 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (decompType .eq. ESMF_GRID_ARBITRARY) then 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 else 
 if (present(totalLWidth) .or. present(totalUWidth)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_WRONG, & 
 msg="- indexflag, totalLWidth or totalUWidth are not allowed for arbitrary grid", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Do General Geom EmptyComplete 
 call ESMF_FieldEmptyComplete(field, geom, farrayPtr, & 
 datacopyflag, gridToFieldMap, & 
 totalLWidth, totalUWidth, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_FieldEmptyCompPtr7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 

!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldEmptyComplete - Complete a Field from Geom started with FieldEmptyCreate 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldEmptyComplete() 
! subroutine ESMF_FieldEmptyCompGB<rank><type><kind>(field, geom, & 
! farray, indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
! ungriddedUBound, totalLWidth, totalUWidth, rc) 
! 
! !ARGUMENTS: 
! type(ESMF_Field), intent(inout) :: field 
! type(ESMF_Geom), intent(in) :: geom 
! <type> (ESMF_KIND_<kind>), dimension(<rank>), target :: farray 
! type(ESMF_Index_Flag), intent(in) :: indexflag 
! type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
! integer, intent(in), optional :: gridToFieldMap(:) 
! integer, intent(in), optional :: ungriddedLBound(:) 
! integer, intent(in), optional :: ungriddedUBound(:) 
! integer, intent(in), optional :: totalLWidth(:) 
! integer, intent(in), optional :: totalUWidth(:) 
! integer, intent(out), optional :: rc 
! 
! !DESCRIPTION: 
! This call completes an {\tt ESMF\_Field} allocated with the 
! {\tt ESMF\_FieldEmptyCreate()} call. For an example and 
! associated documentation using this method see section 
! \ref{sec:field:usage:partial_creation}. 
! 
! The Fortran data pointer inside {\tt ESMF\_Field} can be queried but deallocating 
! the retrieved data pointer is not allowed. 
! 
! The arguments are: 
! \begin{description} 
! \item [field] 
! The {\tt ESMF\_Field} object to be completed and 
! committed in this call. The {\tt field} will have the same dimension 
! (dimCount) as the rank of the {\tt farray}. 
! \item [geom] 
! The {\tt ESMF\_Geom} object to complete the Field. 
! \item [farray] 
! Native Fortran data array to be copied/referenced in the {\tt field}. 
! The {\tt field} dimension (dimCount) will be the same as the dimCount 
! for the {\tt farray}. 
! \item [indexflag] 
! Indicate how DE-local indices are defined. See section 
! \ref{const:indexflag} for a list of valid indexflag options. 
! \item [{[datacopyflag]}] 
! Indicates whether to copy the {\tt farray} or reference it directly. 
! For valid values see \ref{const:datacopyflag}. The default is 
! {\tt ESMF\_DATACOPY\_REFERENCE}. 
! \item [{[gridToFieldMap]}] 
! List with number of elements equal to the 
! {\tt geom}'s dimCount. The list elements map each dimension 
! of the {\tt geom} to a dimension in the {\tt farray} by 
! specifying the appropriate {\tt farray} dimension index. The 
! default is to map all of the {\tt geom}'s dimensions against the 
! lowest dimensions of the {\tt farray} in sequence, i.e. 
! {\tt gridToFieldMap} = (/1,2,3,.../). 
! Unmapped {\tt farray} dimensions are undistributed Field 
! dimensions. 
! All {\tt gridToFieldMap} entries must be greater than or equal 
! to zero and smaller than or equal to the Field dimCount. It is erroneous 
! to specify the same entry multiple times unless it is zero. 
! If the Field dimCount is less than the Grid dimCount then the default 
! gridToFieldMap will contain zeros for the rightmost entries. A zero 
! entry in the {\tt gridToFieldMap} indicates that the particular 
! Grid dimension will be replicating the Field across the DEs along 
! this direction. 
! \item [{[ungriddedLBound]}] 
! Lower bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedLBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt field}. 
! \item [{[ungriddedUBound]}] 
! Upper bounds of the ungridded dimensions of the {\tt field}. 
! The number of elements in the {\tt ungriddedUBound} is equal to the number of ungridded 
! dimensions in the {\tt field}. All ungridded dimensions of the 
! {\tt field} are also undistributed. When field dimension count is 
! greater than grid dimension count, both ungriddedLBound and ungriddedUBound 
! must be specified. When both are specified the values are checked 
! for consistency. Note that the the ordering of 
! these ungridded dimensions is the same as their order in the {\tt field}. 
! \item [{[totalLWidth]}] 
! Lower bound of halo region. The size of this array is the number 
! of gridded dimensions in the {\tt field}. However, ordering of the elements 
! needs to be the same as they appear in the {\tt field}. Values default 
! to 0. If values for totalLWidth are specified they must be reflected in 
! the size of the {\tt field}. That is, for each gridded dimension the 
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[totalUWidth]}] 
! Upper bound of halo region. The size of this array is the number 
! of gridded dimensions in the {\tt field}. However, ordering of the elements 
! needs to be the same as they appear in the {\tt field}. Values default 
! to 0. If values for totalUWidth are specified they must be reflected in 
! the size of the {\tt field}. That is, for each gridded dimension the 
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth} 
! + {\tt computationalCount}, {\tt exclusiveCount} ). 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB1Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 1 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB2Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 2 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB3Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 3 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB4Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 4 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB5Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 5 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB6Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 6 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB7Di1(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 7 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB1Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 1 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB2Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 2 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB3Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 3 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB4Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 4 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB5Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 5 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB6Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 6 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB7Di2(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 7 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB1Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 1 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB1Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 1 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB1Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 1 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB1Dr8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r8), dimension(:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r8), dimension(:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 1 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB2Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 2 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB2Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 2 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB2Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 2 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB2Dr8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r8), dimension(:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 2 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB3Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 3 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB3Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 3 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB3Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 3 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB3Dr8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 3 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB4Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 4 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB4Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 4 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB4Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 4 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB4Dr8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 4 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB5Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 5 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB5Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 5 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB5Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 5 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB5Dr8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 5 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB6Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 6 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB6Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 6 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB6Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 6 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB6Dr8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 6 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB7Di4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 7 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB7Di8(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 7 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! gridToFieldMap elements must be in range 0...fieldRank and unique 
 ! algorithm to check element uniqueness: 
 ! run time: O(ESMF_MAXDIM) 
 ! memory: O(2*ESMF_MAXDIM) 
 ! or O(ESMF_MAXDIM+ESMF_MAXDIM/sizeof(integer)) with bitvector 
 flipflop = .false. 
 do i = 1, gridDimCount 
 if(localGridToFieldMap(i) .lt. 0 .or. & 
 localGridToFieldMap(i) .gt. fieldDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be within range 0...array rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 if(localGridToFieldMap(i) /= 0) then 
 if(flipflop(localGridToFieldMap(i))) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
 msg="- gridToFieldMap element must be unique", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 flipflop(localGridToFieldMap(i)) = .true. 
endif 
 enddo 

 if(present(totalLWidth)) then 
 localMaxHaloLWidth(1:gridDimCount_norep) = & 
 totalLWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloLWidth = 0 
 endif 

 if(present(totalUWidth)) then 
 localMaxHaloUWidth(1:gridDimCount_norep) = & 
 totalUWidth (1:gridDimCount_norep) 
 else 
 localMaxHaloUWidth = 0 
 endif 

 ! Here we get the lbounds and ubounds for ungridded 
 ! dimensions from the native array, if it is not input 
 ! through the argument list. First we need to set up 
 ! an index array that holds the ungridded dimensions of 
 ! the native array. 

 ! Since we are saving the ungriddedIndex calculate it even 
 ! if ungridded bounds are present 

 ! Figure out which dims are ungridded 
 isGridded = .false. 
 do i=1, gridDimCount 
 if(localGridToFieldMap(i) /= 0) isGridded(localGridToFieldMap(i)) = .true. 
 enddo 

 ! Use ungridded info to figure out the map from ungridded to field dims 
 count=1 
 do i=1,fieldDimCount 
 if (.not. isGridded(i)) then 
 ungriddedIndex(count)=i 
 count=count+1 
 endif 
 enddo 

 ! set Array ungridded bounds depending on what user provides
 if (present(ungriddedLBound)) then 
 if(present(ungriddedUBound)) then 
 ! Both present so copy 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! Copy lower bound and make upper bound high enough to fit 
 localUngriddedLBound(1:fieldUngriddedDimCount) = & 
 ungriddedLBound(1:fieldUngriddedDimCount) 
 do i=1, fieldUngriddedDimCount 
 localUngriddedUBound(i) = ungriddedLBound(i)+ & 
 size (farray,ungriddedIndex(i))-1 
 enddo 
 endif 
 else 
 if(present(ungriddedUBound)) then 
 ! Copy upper bound and make lower bound low enough to fit 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = ungriddedUBound(i)- & 
 size (farray,ungriddedIndex(i))+1 
 enddo 
 localUngriddedUBound(1:fieldUngriddedDimCount) = & 
 ungriddedUBound(1:fieldUngriddedDimCount) 
 else 
 ! No user info copy array bounds 
 ! Note: assumed shape bounds will be 1...size 
 do i=1, fieldUngriddedDimCount 
 localUngriddedLBound(i) = lbound(farray,ungriddedIndex(i)) 
 localUngriddedUBound(i) = ubound(farray,ungriddedIndex(i)) 
 enddo 
 endif 
 endif 

 call ESMF_DistGridGet(distgrid, dimCount=distgridDimCount, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 distgridDimCount_norep = memDimCount - fieldUngriddedDimCount 

 ! The undistributed info from the Grid needs to be 
 ! combined with the ungridded info from the Field in order 
 ! to create the Array for the Field. 
 call ESMF_GeomGetArrayInfo(geom, & 
 gridToFieldMap=localGridToFieldMap, & 
 ungriddedLBound=localUngriddedLBound (1:fieldUngriddedDimCount), & 
 ungriddedUBound=localUngriddedUBound (1:fieldUngriddedDimCount), & 
 distgridToArrayMap=distgridToArrayMap, & 
 undistLBound=undistLBound, undistUBound=undistUBound, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Create Array with undistributed dimensions 
 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 totalLWidth=localMaxHaloLWidth(1:gridDimCount_norep), & 
 totalUWidth=localMaxHaloUWidth(1:gridDimCount_norep), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 else 
 array = ESMF_ArrayCreate(distgrid, farray, & 
 indexflag=indexflag, datacopyflag=datacopyflag, & 
 distgridToArrayMap=distgridToArrayMap (1:distgridDimCount), & 
 undistLBound=undistLBound(1:fieldUndistDimCount), & 
 undistUBound=undistUBound(1:fieldUndistDimCount), & 
 name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 field%ftypep%array = array 

 ! set array_internal to .true. because field%array is internal 
 field%ftypep%array_internal = .true. 

 ! Should call a common FieldEmptyCompConstructor here instead 
 ! of just setting things up ourselves 
 ! (The field Sets were all moved here in preparation for this) 
 field%ftypep%gridToFieldMap(1:gridDimCount) = & 
 localGridToFieldMap(1:gridDimCount) 
 field%ftypep%totalLWidth(1:gridDimCount_norep) = & 
 localMaxHaloLWidth (1:gridDimCount_norep) 
 field%ftypep%totalUWidth(1:gridDimCount_norep) = & 
 localMaxHaloUWidth (1:gridDimCount_norep) 
 field%ftypep%ungriddedLBound(1:fieldUngriddedDimCount) = & 
 localUngriddedLBound(1:fieldUngriddedDimCount) 
 field%ftypep%ungriddedUBound(1:fieldUngriddedDimCount) = & 
 localUngriddedUBound(1:fieldUngriddedDimCount) 
 field%ftypep%geom = geom 
 field%ftypep%status = ESMF_FIELDSTATUS_COMPLETE 
 call ESMF_BaseSetStatus(field%ftypep%base, ESMF_STATUS_READY, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 field%ftypep%dimCount = gridDimCount_norep + fieldUngriddedDimCount 

 call ESMF_FieldValidate(field, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(rc)) rc = ESMF_SUCCESS 
end subroutine ESMF_FieldEmptyCompGB7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldEmptyCompGB" 
subroutine ESMF_FieldEmptyCompGB7Dr4(field, geom, farray, & 
 indexflag, datacopyflag, gridToFieldMap, ungriddedLBound, & 
 ungriddedUBound, totalLWidth, totalUWidth, rc) 

 ! input arguments 
 type(ESMF_Field), intent(inout) :: field 
 type(ESMF_Geom), intent(in) :: geom 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), target :: farray 
 type(ESMF_Index_Flag), intent(in) :: indexflag 
 type(ESMF_DataCopy_Flag), intent(in), optional :: datacopyflag 
 integer, intent(in), optional :: gridToFieldMap(:) 
 integer, intent(in), optional :: ungriddedLBound(:) 
 integer, intent(in), optional :: ungriddedUBound(:) 
 integer, intent(in), optional :: totalLWidth(:) 
 integer, intent(in), optional :: totalUWidth(:) 
 integer, intent(out), optional :: rc 
 ! local variables 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: fpointer 
 integer :: localrc, i, j, count 
 integer :: memDimCount, fieldUngriddedDimCount 
 integer :: fieldDimCount, localDeCount 
 integer :: gridDimCount, gridDimCount_norep 
 integer :: grid_repdimcount 
 integer :: elementCount 
 integer :: ungriddedIndex(ESMF_MAXDIM) 
 integer :: distgridToArrayMap (ESMF_MAXDIM) 
 integer :: undistLBound(ESMF_MAXDIM), undistUBound(ESMF_MAXDIM) 
 integer :: localUngriddedLBound (ESMF_MAXDIM) 
 integer :: localUngriddedUBound (ESMF_MAXDIM) 
 integer :: localGridToFieldMap (ESMF_MAXDIM) 
 integer :: localMaxHaloLWidth (ESMF_MAXDIM) 
 integer :: localMaxHaloUWidth (ESMF_MAXDIM) 
 logical :: isGridded(ESMF_MAXDIM) 
 integer :: distgridToGridMap(ESMF_MAXDIM) 
 type(ESMF_Array) :: array 
 type(ESMF_DistGrid) :: distgrid 
 integer :: fieldUndistDimCount 
 logical :: flipflop(ESMF_MAXDIM) 
 type(ESMF_Index_Flag) :: g_indexflag 
 logical :: found 
 type(ESMF_GridDecompType) :: decompType 
 type(ESMF_GeomType_Flag) :: geomtype 
 type(ESMF_Grid) :: grid 
 integer :: distgridDimCount, distgridDimCount_norep, arbdim 
 character(len=ESMF_MAXSTR) :: fieldName 

 if (present(rc)) then 
 rc = ESMF_RC_NOT_IMPL 
 endif 
 localrc = ESMF_RC_NOT_IMPL 

 ! make sure field, grid, farray are properly initialized 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 
 ESMF_INIT_CHECK_DEEP(ESMF_GeomGetInit,geom,rc) 

 ! get the fieldName 
 call ESMF_FieldGet(field, name=fieldName, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 ! Get number of grid dimensions, number 
 ! of distributed grid dimensions, distgrid, 
 ! number of ungridded Field dimensions, 
 ! and number of undistributed Field Dimensions 
 call ESMF_GeomGet(geom, dimCount=gridDimCount, & 
 distgridToGridMap=distgridToGridMap, localDeCount=localDeCount, & 
 distgrid=distgrid, indexflag=g_indexflag, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if(.not. (indexflag .eq. g_indexflag)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- user specified indexflag must be identical with Grid indexflag", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 elementCount = 0 ! default to assuming no elements 
 if (localDeCount==1) then 
 call ESMF_DistGridGet(distgrid, localDe=0, elementCount=elementCount, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 elseif (localDeCount>1) then 
 !TODO: error out because not supported 
 endif 

 if (elementCount > 0) then 
 ! The following use of fptr is a bit of trickery to get all F90 
 ! compilers to cooperate. For some compilers the associated() test 
 ! will return .false. for farray of size 0. Some of those compilers 
 ! will produce a run-time error in size(fptr). Other compilers will 
 ! return .true. for the associated() test but return 0 in size(). 
 fpointer => farray 
 if(.not. associated(fpointer,farray)) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 

 if (size(fpointer)==0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, & 
 msg="- farray is not associated with memory allocation)", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 ! Check if geom is a grid, if so, check if it is arbitrary 
 decompType = ESMF_GRID_NONARBITRARY 
 call ESMF_GeomGet(geom, geomtype=geomtype, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (geomtype .eq. ESMF_GEOMTYPE_GRID) then 
 call ESMF_GeomGet(geom, grid=grid, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 call ESMF_GridGetDecompType(grid, decompType, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 endif 

 ! Check the size of the native array. 
 memDimCount = 7 

 ! Error Check Input 
 grid_repdimcount = 0 
 if (present(gridToFieldMap)) then 
 if (size(gridToFieldMap) .ne. gridDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- gridToFieldMap size must equal to grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 do i = 1, gridDimCount 
 if(gridToFieldMap(i) == 0) grid_repdimcount = grid_repdimcount + 1 
 enddo 
 endif 
 gridDimCount_norep = gridDimCount - grid_repdimcount 

 if (present(gridToFieldMap)) then 
 localGridToFieldMap(1:gridDimCount) = & 
 gridToFieldMap (1:gridDimCount) 
 else 
 do i = 1, gridDimCount 
 localGridToFieldMap(i) = i 
 enddo 
 endif 

 if (decompType .eq. ESMF_GRID_NONARBITRARY) then 
 fieldDimCount = memDimCount 
 else 
 call ESMF_GridGet(grid, distgridToGridMap=distgridToGridMap, & 
 rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 ! find out how many grid dimensions are arbitrarily distributed and calculate 
 ! fieldDimCount accordingly 
 arbdim = 0 
 do i=1,gridDimCount 
 if (distgridToGridMap(i) .ne. 0) arbdim = arbdim+1 
 enddo 
 fieldDimCount = memDimCount + arbdim - 1 

 ! If there is any replicated dimension, check if any of the arb. dimensions are replicated. 
 ! If one arb dimension is replicated, all the arb. dimensions have to be replicated 
 if (grid_repdimcount .ne. 0) then 
 do i = 1,gridDimCount 
 if(localGridToFieldMap(i) == 0) then 
 found = .false. 
 do j=1,arbdim 
 if (distgridToGridMap(j) .eq. i) found = .true. 
 enddo 
 if (found) then 
 ! one arb.dimension is a replicated dimension, check if other arb dimensions are 
 ! also replicated 
 do j=1,arbdim 
 if (distgridToGridMap(j) .ne. i) then 
 if (localGridToFieldMap(distgridToGridMap(j)) .ne. 0) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- Arb. grid dimensions have to be either all replicated or not replicated", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 
 enddo 
 ! all arb. dimension are replication, jump out of the first do loop 
 ! fieldDimCount should be the same as the memDimCount 
 fieldDimCount = memDimCount 
 exit 
 endif 
 endif 
 enddo 
 endif 
 endif 

 if(fieldDimCount .lt. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- farray rank must be greater than or equal to grid rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
! refer to ticket 1888180 
! if(fieldDimCount .gt. gridDimCount_norep) then 
! if( (.not. present(ungriddedLBound)) .or. & 
! (.not. present(ungriddedUBound)) ) then 
! call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, & 
! "- ungridded bounds must be present when Field has ungridded dimension(s)", & 
! ESMF_CONTEXT, rcToReturn=rc) 
! return 
! endif 
! endif 

 ! ungridded dimensions plus replicated dimensions 
 fieldUngriddedDimCount = fieldDimCount-gridDimCount + grid_repdimcount 
 fieldUndistDimCount = fieldDimCount-gridDimCount + grid_repdimcount 

 ! Error Check Input 
 if (present(ungriddedLBound)) then 
 if (size(ungriddedLBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedLBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(ungriddedUBound)) then 
 if (size(ungriddedUBound) .ne. fieldUngriddedDimCount) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- ungriddedUBound size must equal to array_rank-grid_rank", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalLWidth)) then 
 if (size(totalLWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalLWidth must equal to gridded dimCount", & 
 ESMF_CONTEXT, rcToReturn=rc) 
 return 
 endif 
 endif 

 if (present(totalUWidth)) then 
 if (size(totalUWidth) .ne. gridDimCount_norep) then 
 call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_SIZE, & 
 msg="- totalUWidth must equal to gridded dimC