//===-- CommandLineOpts.h -- shared command line options --------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

/// This file declares some shared command-line options that can be used when
/// debugging the test tools.

#ifndef FORTRAN_OPTIMIZER_PASSES_COMMANDLINEOPTS_H
#define FORTRAN_OPTIMIZER_PASSES_COMMANDLINEOPTS_H

#include "llvm/Frontend/Debug/Options.h"
#include "llvm/Passes/OptimizationLevel.h"
#include "llvm/Support/CommandLine.h"

/// Shared option in tools to control whether dynamically sized array
/// allocations should always be on the heap.
extern llvm::cl::opt<bool> dynamicArrayStackToHeapAllocation;

/// Shared option in tools to set a maximum value for the number of elements in
/// a compile-time sized array that can be allocated on the stack.
extern llvm::cl::opt<std::size_t> arrayStackAllocationThreshold;

/// Shared option in tools to ignore missing runtime type descriptor objects
/// when translating FIR to LLVM. The resulting program will crash if the
/// runtime needs the derived type descriptors, this is only a debug option to
/// allow compiling manually written FIR programs involving derived types
/// without having to write the derived type descriptors which are normally
/// generated by the frontend.
extern llvm::cl::opt<bool> ignoreMissingTypeDescriptors;

/// Shared option in tools to only generate rtti static object definitions for
/// derived types defined in the current compilation unit. Derived type
/// descriptor object for types defined in other objects will only be declared
/// as external. This also changes the linkage of rtti objects defined in the
/// current compilation unit from linkonce_odr to external so that unused rtti
/// objects are retained and can be accessed from other compilation units. This
/// is an experimental option to explore compilation speed improvements and is
/// an ABI breaking change because of the linkage change.
/// It will also require linking against module file objects of modules defining
/// only types (even for trivial types without type bound procedures, which
/// differs from most compilers).
extern llvm::cl::opt<bool> skipExternalRttiDefinition;

/// Default optimization level used to create Flang pass pipeline is O0.
extern llvm::OptimizationLevel defaultOptLevel;

extern llvm::codegenoptions::DebugInfoKind noDebugInfo;

/// Optimizer Passes
extern llvm::cl::opt<bool> disableCfgConversion;
extern llvm::cl::opt<bool> disableFirAliasTags;
extern llvm::cl::opt<bool> disableFirAvc;
extern llvm::cl::opt<bool> disableFirMao;
extern llvm::cl::opt<bool> enableFirLICM;
extern llvm::cl::opt<bool> useOldAliasTags;

/// CodeGen Passes
extern llvm::cl::opt<bool> disableCodeGenRewrite;
extern llvm::cl::opt<bool> disableTargetRewrite;
extern llvm::cl::opt<bool> disableDebugInfo;
extern llvm::cl::opt<bool> disableFirToLlvmIr;
extern llvm::cl::opt<bool> disableLlvmIrToLlvm;
extern llvm::cl::opt<bool> disableBoxedProcedureRewrite;

extern llvm::cl::opt<bool> disableExternalNameConversion;
extern llvm::cl::opt<bool> enableConstantArgumentGlobalisation;
extern llvm::cl::opt<bool> disableCompilerGeneratedNamesConversion;

#endif // FORTRAN_OPTIMIZER_PASSES_COMMANDLINE_OPTS_H
