/*
 * Library to support cross-platform C string functions
 *
 * Copyright (C) 2010-2015, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#if !defined( _LIBCSTRING_H )
#define _LIBCSTRING_H

#include <libcstring/definitions.h>
#include <libcstring/extern.h>
#include <libcstring/features.h>
#include <libcstring/types.h>

#if defined( __cplusplus )
extern "C" {
#endif

/* -------------------------------------------------------------------------
 * Support functions
 * ------------------------------------------------------------------------- */

/* Returns the library version as a string
 */
LIBCSTRING_EXTERN \
const char *libcstring_get_version(
             void );

/* -------------------------------------------------------------------------
 * Narrow character string functions
 * ------------------------------------------------------------------------- */

/* String allocation
 */
#if defined( HAVE_GLIB_H )
#define libcstring_narrow_string_allocate( size ) \
	g_malloc( (gsize) ( sizeof( char ) * size ) )

#elif defined( HAVE_MALLOC ) || defined( WINAPI )
#define libcstring_narrow_string_allocate( size ) \
	malloc( sizeof( char ) * size )

#elif defined( WINAPI )
#define libcstring_narrow_string_allocate( size ) \
	HeapAlloc( GetProcessHeap(), 0, (SIZE_T) ( sizeof( char ) * size ) )
#endif

/* String reallocation
 */
#if defined( HAVE_GLIB_H )
#define libcstring_narrow_string_reallocate( buffer, size ) \
	g_realloc( (gpointer) buffer, (gsize) ( sizeof( char ) * size ) )

#elif defined( HAVE_REALLOC ) || defined( WINAPI )
#define libcstring_narrow_string_reallocate( buffer, size ) \
	realloc( (void *) buffer, ( sizeof( char ) * size ) )

#elif defined( WINAPI )
/* HeapReAlloc does not allocate empty (NULL) buffers as realloc does
 */
#define libcstring_narrow_string_reallocate( buffer, size ) \
	( buffer == NULL ) ? \
	HeapAlloc( GetProcessHeap(), 0, (SIZE_T) ( sizeof( char ) * size ) ) : \
	HeapReAlloc( GetProcessHeap(), 0, (LPVOID) buffer, (SIZE_T) ( sizeof( char ) * size ) )
#endif

/* String length
 */
#if defined( HAVE_STRLEN ) || defined( WINAPI )
#define libcstring_narrow_string_length( string ) \
	strlen( string )
#endif

/* String compare
 */
#if defined( HAVE_MEMCMP ) || defined( WINAPI )
#define libcstring_narrow_string_compare( string1, string2, size ) \
	memcmp( (void *) string1, (void *) string2, size )

#elif defined( HAVE_STRNCMP )
#define libcstring_narrow_string_compare( string1, string2, size ) \
	strncmp( string1, string2, size )
#endif

/* Caseless string compare
 */
#if defined( HAVE_GLIB_H )
#define libcstring_narrow_string_compare_no_case( string1, string2, size ) \
	g_ascii_strncasecmp( string1, string2, size )

#elif defined( _MSC_VER )
#define libcstring_narrow_string_compare_no_case( string1, string2, size ) \
	_strnicmp( string1, string2, size )

#elif defined( WINAPI )
#define libcstring_narrow_string_compare_no_case( string1, string2, size ) \
	strnicmp( string1, string2, size )

#elif defined( HAVE_STRNCASECMP )
#define libcstring_narrow_string_compare_no_case( string1, string2, size ) \
	strncasecmp( string1, string2, size )

#elif defined( HAVE_STRCASECMP )
#define libcstring_narrow_string_compare_no_case( string1, string2, size ) \
	strcasecmp( string1, string2 )
#endif

/* String copy
 */
#if defined( HAVE_MEMCPY ) || defined( WINAPI )
#define libcstring_narrow_string_copy( destination, source, size ) \
	(char *) memcpy( (void *) destination, (void *) source, size )

#elif defined( HAVE_STRNCPY )
#define libcstring_narrow_string_copy( destination, source, size ) \
	strncpy( destination, source, size )
#endif

/* String character search
 */
#if defined( HAVE_MEMCHR ) || defined( WINAPI )
#define libcstring_narrow_string_search_character( string, character, size ) \
	(char *) memchr( (void *) string, (int) character, size )

#elif defined( HAVE_STRCHR )
#define libcstring_narrow_string_search_character( string, character, size ) \
	strchr( string, (int) character )
#endif

/* String reverse character search
 */
#if defined( HAVE_MEMRCHR ) && ( HAVE_DECL_MEMRCHR == 1 )
#define libcstring_narrow_string_search_character_reverse( string, character, size ) \
	(char *) memrchr( (void *) string, (int) character, size )

#elif defined( HAVE_STRRCHR ) || defined( WINAPI )
#define libcstring_narrow_string_search_character_reverse( string, character, size ) \
	strrchr( string, (int) character )
#endif

/* String sub-string search
 */
#if defined( HAVE_STRSTR ) || defined( WINAPI )
#define libcstring_narrow_string_search_string( string, substring, size ) \
	strstr( string, substring )
#endif

/* String formatted print (snprintf)
 */
#if defined( HAVE_GLIB_H )
#define libcstring_narrow_string_snprintf( target, size, ... ) \
	g_snprintf( target, size, __VA_ARGS__ )

#elif defined( _MSC_VER )
#define libcstring_narrow_string_snprintf( target, size, ... ) \
	sprintf_s( target, size, __VA_ARGS__ )

#elif defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_narrow_string_snprintf \
	snprintf

#elif defined( HAVE_SNPRINTF ) || defined( WINAPI )
#define libcstring_narrow_string_snprintf( target, size, ... ) \
	snprintf( target, size, __VA_ARGS__ )
#endif

/* String input conversion (sscanf)
 */
#if defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_narrow_string_sscanf \
	sscanf

#elif defined( HAVE_SSCANF ) || defined( WINAPI )
#define libcstring_narrow_string_sscanf( string, format, ... ) \
	sscanf( string, format, __VA_ARGS__ )
#endif

/* Variable arguments formatted print to string function (vsnprintf)
 */
#if defined( HAVE_GLIB_H )
#define libcstring_narrow_string_vsnprintf( string, size, format, ... ) \
	g_vsnprintf( string, size, format, __VA_ARGS__ )

#elif defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_narrow_string_vsnprintf \
	vsnprintf

#elif defined( HAVE_VSNPRINTF ) || defined( WINAPI )
#define libcstring_narrow_string_vsnprintf( string, size, format, ... ) \
	vsnprintf( string, size, format, __VA_ARGS__ )
#endif

/* -------------------------------------------------------------------------
 * Wide character string functions
 * ------------------------------------------------------------------------- */

/* String allocation
 */
#if defined( HAVE_GLIB_H )
#define libcstring_wide_string_allocate( size ) \
	g_malloc( (gsize) ( sizeof( wchar_t ) * size ) )

#elif defined( HAVE_MALLOC ) || defined( WINAPI )
#define libcstring_wide_string_allocate( size ) \
	malloc( sizeof( wchar_t ) * size )

#elif defined( WINAPI )
#define libcstring_wide_string_allocate( size ) \
	HeapAlloc( GetProcessHeap(), 0, (SIZE_T) ( sizeof( wchar_t ) * size ) )
#endif

/* String reallocation
 */
#if defined( HAVE_GLIB_H )
#define libcstring_wide_string_reallocate( buffer, size ) \
	g_realloc( (gpointer) buffer, (gsize) ( sizeof( wchar_t ) * size ) )

#elif defined( HAVE_REALLOC ) || defined( WINAPI )
#define libcstring_wide_string_reallocate( buffer, size ) \
	realloc( (void *) buffer, ( sizeof( wchar_t ) * size ) )

#elif defined( WINAPI )
/* HeapReAlloc does not allocate empty (NULL) buffers as realloc does
 */
#define libcstring_wide_string_reallocate( buffer, size ) \
	( buffer == NULL ) ? \
	HeapAlloc( GetProcessHeap(), 0, (SIZE_T) ( sizeof( wchar_t ) * size ) ) : \
	HeapReAlloc( GetProcessHeap(), 0, (LPVOID) buffer, (SIZE_T) ( sizeof( wchar_t ) * size ) )
#endif

/* String length
 */
#if defined( HAVE_WCSLEN ) || defined( WINAPI )
#define libcstring_wide_string_length( string ) \
	wcslen( string )
#endif

/* String compare
 */
#if defined( HAVE_WMEMCMP )
#define libcstring_wide_string_compare( string1, string2, size ) \
	wmemcmp( (void *) string1, (void *) string2, size )

#elif defined( HAVE_WCSNCMP ) || defined( WINAPI )
#define libcstring_wide_string_compare( string1, string2, size ) \
	wcsncmp( string1, string2, size )
#endif

/* Caseless string compare
 */
#if defined( _MSC_VER ) || ( defined( __BORLANDC__ ) && ( __BORLANDC__ >= 0x0551 ) )
#define libcstring_wide_string_compare_no_case( string1, string2, size ) \
	_wcsnicmp( string1, string2, size )

#elif defined( WINAPI )
#define libcstring_wide_string_compare_no_case( string1, string2, size ) \
	wcsnicmp( string1, string2, size )

#elif defined( HAVE_WCSNCASECMP )
#define libcstring_wide_string_compare_no_case( string1, string2, size ) \
	wcsncasecmp( string1, string2, size )

#elif defined( HAVE_WCSCASECMP )
#define libcstring_wide_string_compare_no_case( string1, string2, size ) \
	wcscasecmp( string1, string2 )

#elif defined( HAVE_TOWLOWER )
LIBCSTRING_EXTERN \
int libcstring_wide_string_compare_no_case(
     const wchar_t *string1,
     const wchar_t *string2,
     size_t comparision_length );

#endif

/* String copy
 */
#if defined( HAVE_WMEMCPY )
#define libcstring_wide_string_copy( destination, source, size ) \
	(wchar_t *) wmemcpy( (void *) destination, (void *) source, size )

#elif defined( HAVE_WCSNCPY ) || defined( WINAPI )
#define libcstring_wide_string_copy( destination, source, size ) \
	wcsncpy( destination, source, size )
#endif

/* String character search
 */
#if defined( HAVE_WMEMCHR )
#define libcstring_wide_string_search_character( string, character, size ) \
	(wchar_t *) wmemchr( (void *) string, (wchar_t) character, size )

#elif defined( HAVE_WCSCHR ) || defined( WINAPI )
#define libcstring_wide_string_search_character( string, character, size ) \
	wcschr( string, (wchar_t) character )

#endif

/* String reverse character search
 */
#if defined( HAVE_WMEMRCHR )
#define libcstring_wide_string_search_character_reverse( string, character, size ) \
	(wchar_t *) wmemrchr( (void *) string, (wchar_t) character, size )

#elif defined( HAVE_WCSRCHR ) || defined( WINAPI )
#define libcstring_wide_string_search_character_reverse( string, character, size ) \
	wcsrchr( string, (wchar_t) character )
#endif

/* String sub-string search
 */
#if defined( HAVE_WCSSTR ) || defined( WINAPI )
#define libcstring_wide_string_search_string( string, substring, size ) \
	wcsstr( string, substring )

#endif

/* String formatted print (snwprintf)
 */
#if defined( _MSC_VER )
#define libcstring_wide_string_snwprintf( target, size, ... ) \
	swprintf_s( target, size, __VA_ARGS__ )

#elif defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_wide_string_snwprintf \
	snwprintf

#elif defined( WINAPI )
#define libcstring_wide_string_snwprintf( target, size, ... ) \
	snwprintf( target, size, __VA_ARGS__ )

#elif defined( HAVE_SWPRINTF )
#define libcstring_wide_string_snwprintf( target, size, ... ) \
	swprintf( target, size, __VA_ARGS__ )
#endif

/* Variable arguments formatted print to string function (vsnwprintf)
 */
#if defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_wide_string_vsnwprintf \
	_vsnwprintf

#elif defined( WINAPI )
#define libcstring_wide_string_vsnwprintf( string, size, format, ... ) \
	_vsnwprintf( string, size, format, __VA_ARGS__ )

#elif defined( HAVE_VSWPRINTF )
#define libcstring_wide_string_vsnwprintf( string, size, format, ... ) \
	vswprintf( string, size, format, __VA_ARGS__ )
#endif

/* -------------------------------------------------------------------------
 * System (character) string functions
 * ------------------------------------------------------------------------- */

LIBCSTRING_EXTERN \
int libcstring_narrow_system_string_codepage;

#if defined( LIBCSTRING_HAVE_WIDE_SYSTEM_CHARACTER )

#if SIZEOF_WCHAR_T != 2
#error Unsupported wide system character size
#endif

/* Intermediate version of the macro required
 * for correct evaluation predefined string
 */
#define _LIBCSTRING_SYSTEM_STRING_INTERMEDIATE( string ) \
	L ## string

#define _LIBCSTRING_SYSTEM_STRING( string ) \
	_LIBCSTRING_SYSTEM_STRING_INTERMEDIATE( string )

#define libcstring_system_string_allocate( size ) \
	libcstring_wide_string_allocate( size )

#define libcstring_system_string_reallocate( string, size ) \
	libcstring_wide_string_reallocate( string, size )

#define libcstring_system_string_compare( destination, source, size ) \
	libcstring_wide_string_compare( destination, source, size )

#define libcstring_system_string_compare_no_case( destination, source, size ) \
	libcstring_wide_string_compare_no_case( destination, source, size )

#define libcstring_system_string_copy( destination, source, size ) \
	libcstring_wide_string_copy( destination, source, size )

#define libcstring_system_string_length( string ) \
	libcstring_wide_string_length( string )

#define libcstring_system_string_search_character( string, character, size ) \
	libcstring_wide_string_search_character( string, character, size )

#define libcstring_system_string_search_character_reverse( string, character, size ) \
	libcstring_wide_string_search_character_reverse( string, character, size )

#define libcstring_system_string_search_string( string, substring, size ) \
	libcstring_wide_string_search_string( string, substring, size )

#if defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_system_string_sprintf \
	libcstring_wide_string_snwprintf

#else
#define libcstring_system_string_sprintf( string, size, format, ... ) \
	libcstring_wide_string_snwprintf( string, size, format, __VA_ARGS__ )
#endif

#if defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_system_string_vsprintf \
	libcstring_wide_string_vsnwprintf

#else
#define libcstring_system_string_vsprintf( string, size, format, ... ) \
	libcstring_wide_string_vsnwprintf( string, size, format, __VA_ARGS__ )
#endif

#else

#define _LIBCSTRING_SYSTEM_STRING( string ) \
	string

#define libcstring_system_string_allocate( size ) \
	libcstring_narrow_string_allocate( size )

#define libcstring_system_string_reallocate( string, size ) \
	libcstring_narrow_string_reallocate( string, size )

#define libcstring_system_string_compare( destination, source, size ) \
	libcstring_narrow_string_compare( destination, source, size )

#define libcstring_system_string_compare_no_case( destination, source, size ) \
	libcstring_narrow_string_compare_no_case( destination, source, size )

#define libcstring_system_string_copy( destination, source, size ) \
	libcstring_narrow_string_copy( destination, source, size )

#define libcstring_system_string_length( string ) \
	libcstring_narrow_string_length( string )

#define libcstring_system_string_search_character( string, character, size ) \
	libcstring_narrow_string_search_character( string, character, size )

#define libcstring_system_string_search_character_reverse( string, character, size ) \
	libcstring_narrow_string_search_character_reverse( string, character, size )

#define libcstring_system_string_search_string( string, substring, size ) \
	libcstring_narrow_string_search_string( string, substring, size )

#if defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_system_string_sprintf \
	libcstring_narrow_string_snprintf

#else
#define libcstring_system_string_sprintf( string, size, format, ... ) \
	libcstring_narrow_string_snprintf( string, size, format, __VA_ARGS__ )
#endif

#if defined( __BORLANDC__ ) && ( __BORLANDC__ < 0x0560 )
#define libcstring_system_string_vsprintf \
	libcstring_narrow_string_vsnprintf

#else
#define libcstring_system_string_vsprintf( string, size, format, ... ) \
	libcstring_narrow_string_vsnprintf( string, size, format, __VA_ARGS__ )
#endif

#endif /* defined( LIBCSTRING_HAVE_WIDE_SYSTEM_CHARACTER ) */

#if defined( __cplusplus )
}
#endif

#endif

