#ifndef _LOCALIZER_H_
#define _LOCALIZER_H_

/** @file localizer.h */

#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef SWIG
%module Localizer
%{
#include "localizer.h"
%}
#endif

/**
 * \mainpage Localizer API Documenation
 * 
 * \section Introduction
 * 
 * The Localizer API provides an interface to the localizer IP database. The database contains the 
 * relations between Internet Adresses (IPv4) and the corresponding geographical position.
 * 
 * The API is written in C and is the base for many other script-language interfaces like the PHP-API
 * and the Perl-API.
 * 
 * It provides the necessary function to create get a connection to database and search for a dataset 
 * specified by the IP-address.
 */

#ifndef LOCALIZER_INTERNAL
typedef void localizer;
#endif

typedef struct {
	unsigned long netmask; /**< netmask */
	unsigned long hostmask; /**< hostmask */
	
	const char *city;
	const char *province;
	const char *country;
	const char *provider;
	
	double distance; /**< estimated distance to the nearest access-point */
	double latitude; /**< latitude */
	double longitude; /**< longitude */
} l_data_export;

typedef struct {
	int    num_elements;
	
	/* shm */
	int    shm_enabled;
	int    shm_size; /**< calculated size of the share-memory segment */
	time_t shm_last_mod; /**< last modification time for the database (.csv) */
	off_t  shm_db_size; /**< original size of the database (.cvs) */
} l_status_export;


/**
 * convert a IP in dot-notation into the internal representation (long int)
 * 
 * localizer_search() requires a 
 * 
 * @param ip IP in dot-notation (e.g. 192.168.23.3)
 * @return IP in long int represenation, 0 on error
 * @see localizer_search
 */
long localizer_ip2int(const char *ip );

/**
 * lookup a ip in the database
 * 
 * @param loc localizer handle
 * @param ip IP in long int represenation
 * @return index to the loc->ndat array
 * @see localizer_ip2int
 */
int localizer_search (const localizer *loc, unsigned long ip, l_data_export *data);
int localizer_status (const localizer *loc, l_status_export *data);

/**
 * read the CSV database from the file named <filename>
 * 
 * if filename is NULL stdin is used
 * 
 * @param l localizer handle
 * @param filename name of the database
 * 
 */
int localizer_read(localizer *l, const char *filename);

/**
 * read the CSV database from the file named <filename>
 * 
 * if filename is NULL a existing localizer-db in shm is used
 * 
 * @param l localizer handle
 * @param filename name of the database
 * 
 */
int localizer_read_shm(localizer *l, const char *filename);

/**
 * create a localizer handle
 * 
 * @return localizer handle
 */
localizer * localizer_init();

/**
 * destroy a localizer handle generated by localizer_init
 * 
 * @param l localizer handle
 * @see localizer_init
 */
void localizer_free(localizer * l);

#endif
