import { Thread, type ThreadOptions } from '../Thread.ts';
export type ServiceWorkerMessageSource = NonNullable<ExtendableMessageEvent['source']>;
export interface ThreadServiceWorkerClientsOptions<Imports = Record<string, never>, Exports = Record<string, never>> extends ThreadOptions<Imports, Exports> {
    include?(source: ServiceWorkerMessageSource): boolean;
}
/**
 * Starts an object within a service worker that will listen for new clients connecting,
 * and immediately create a thread around each client. You can then call the `get()`, `create()`,
 * and `delete()` methods to retrieve the threads for a given client.
 *
 * @example
 * import {ThreadServiceWorkerClients} from '@quilted/threads';
 *
 * const threads = new ThreadServiceWorkerClients();
 *
 * addEventListener('message', (event) => {
 *   const source = event.source;
 *   const thread = threads.get(source);
 *   const message = await thread.imports.getMessage();
 * });
 */
export declare class ThreadServiceWorkerClients<Imports = Record<string, never>, Exports = Record<string, never>> {
    #private;
    /**
     * Starts a listening for new clients connecting to the service worker, and
     * creates a thread around each, with the second argument as the exports of the thread.
     *
     * @example
     * ```ts
     * import {ThreadServiceWorker} from '@quilted/threads';
     *
     * // In your service worker:
     *
     * import {ThreadServiceWorkerClients} from '@quilted/threads';
     *
     * ThreadServiceWorkerClients.export({
     *   async getMessage() {
     *     return 'Hello, world!';
     *   },
     * });
     *
     * // On the main thread:
     *
     * const registration = await navigator.serviceWorker.register('worker.js');
     * const serviceWorker = registration.installing ?? registration.waiting ?? registration.active;
     * const {getMessage} = ThreadServiceWorker.import(serviceWorker);
     * const message = await getMessage(); // 'Hello, world!'
     * ```
     */
    static export<Exports = Record<string, never>>(exports: Exports, options?: Omit<ThreadServiceWorkerClientsOptions<Record<string, never>, Exports>, 'exports'>): void;
    constructor(options?: ThreadServiceWorkerClientsOptions<Imports, Exports>);
    get(client: ServiceWorkerMessageSource): Thread<Imports, Exports> | undefined;
    delete(client: ServiceWorkerMessageSource): boolean;
    from(client: ServiceWorkerMessageSource, overrideOptions?: ThreadOptions<Imports, Exports>): Thread<Imports, Exports>;
    export(client: ServiceWorkerMessageSource, exports: Exports, overrideOptions?: Omit<ThreadOptions<Imports, Exports>, 'exports' | 'imports'>): void;
    import(client: ServiceWorkerMessageSource, overrideOptions?: Omit<ThreadOptions<Imports, Exports>, 'exports'>): import("../Thread.ts").ThreadImports<Imports>;
    /** @deprecated Use `from()` instead. */
    create(client: ServiceWorkerMessageSource, overrideOptions?: ThreadOptions<Imports, Exports>): Thread<Imports, Exports>;
}
//# sourceMappingURL=ThreadServiceWorkerClients.d.ts.map