// Copyright (c) 2020-now by the Zeek Project. See LICENSE for details.

#pragma once

#include <string>
#include <utility>

#include <hilti/ast/ctor.h>
#include <hilti/ast/types/string.h>

namespace hilti::ctor {

/** AST node for a `string` ctor. */
class String : public Ctor {
public:
    const auto& value() const { return _value; }
    auto isLiteral() const { return _is_literal; }

    QualifiedType* type() const final { return child<QualifiedType>(0); }

    node::Properties properties() const final {
        auto p = node::Properties{{"value", _value}, {"is_literal", _is_literal}};
        return Ctor::properties() + std::move(p);
    }

    static auto create(ASTContext* ctx, std::string value, bool is_literal, const Meta& meta = {}) {
        return ctx->make<String>(ctx, {QualifiedType::create(ctx, type::String::create(ctx, meta), Constness::Const)},
                                 std::move(value), is_literal, meta);
    }

protected:
    String(ASTContext* ctx, Nodes children, std::string value, bool is_literal, Meta meta)
        : Ctor(ctx, NodeTags, std::move(children), std::move(meta)),
          _value(std::move(value)),
          _is_literal(is_literal) {}

    HILTI_NODE_1(ctor::String, Ctor, final);

private:
    std::string _value;
    bool _is_literal = false;
};

} // namespace hilti::ctor
